<?php
use Adianti\Control\TAction;
use Adianti\Control\TPage;
use Adianti\Widget\Container\TPanelGroup;
use Adianti\Widget\Dialog\TMessage;
use Adianti\Widget\Form\TEntry;
use Adianti\Database\TRepository;
use Adianti\Database\TCriteria;
use Adianti\Database\TFilter;
use Adianti\Widget\Wrapper\TQuickForm;
use Adianti\Widget\Datagrid\TDataGridColumn;
use Adianti\Widget\Datagrid\TDataGridAction;
use Adianti\Widget\Datagrid\TDataGrid;
use Adianti\Widget\Dialog\TQuestion;

use Adianti\Database\TTransaction;

// Compat: alias para TSession em diferentes versões
if (!class_exists('TSession')) {
    if (class_exists('\Adianti\Core\TSession')) {
        class_alias('\Adianti\Core\TSession', 'TSession');
    } elseif (class_exists('\Adianti\Registry\TSession')) {
        class_alias('\Adianti\Registry\TSession', 'TSession');
    }
}

class LogradouroList extends TPage
{
    private $form;
    private $datagrid;
    private $loaded;

    public function __construct()
    {
        parent::__construct();

        $this->form = new TQuickForm('form_search_Logradouro');
        $this->form->setFormTitle('Logradouro - Busca');

        $id   = new TEntry('id');
        $nome = new TEntry('nome');
        $cep  = new TEntry('cep_logradouro');

        $this->form->addQuickField('ID',  $id,   100);
        $this->form->addQuickField('Nome',$nome, 300);
        $this->form->addQuickField('CEP', $cep,  150);

        $this->form->setData( TSession::getValue('search_Logradouro') );

        $this->form->addQuickAction('Buscar', new TAction([$this, 'onSearch']), 'fa:search');
        $this->form->addQuickAction('Novo',   new TAction(['LogradouroForm', 'onClear']), 'fa:plus green');

        $this->datagrid = new TDataGrid;
        $this->datagrid->style     = 'width: 100%';
        $this->datagrid->datatable = 'true';

        $col_id           = new TDataGridColumn('id', 'ID', 'center', '6%');
        $col_tipo_nome    = new TDataGridColumn('tipo_nome', 'Tipo', 'left', '14%');
        $col_nome         = new TDataGridColumn('nome', 'Nome', 'left', '36%');
        $col_cep          = new TDataGridColumn('cep_logradouro', 'CEP', 'center', '12%');
        $col_bairro_ini   = new TDataGridColumn('bairro_inicial_nome', 'Bairro Inicial', 'left', '16%');
        $col_bairro_fim   = new TDataGridColumn('bairro_final_nome', 'Bairro Final', 'left', '16%');

        $this->datagrid->addColumn($col_id);
        $this->datagrid->addColumn($col_tipo_nome);
        $this->datagrid->addColumn($col_nome);
        $this->datagrid->addColumn($col_cep);
        $this->datagrid->addColumn($col_bairro_ini);
        $this->datagrid->addColumn($col_bairro_fim);

        $action_edit = new TDataGridAction(['LogradouroForm', 'onEdit']);
        $action_edit->setLabel('Editar');
        $action_edit->setImage('fa:edit blue');
        $action_edit->setField('id');

        $action_del  = new TDataGridAction([$this, 'onDelete']);
        $action_del->setLabel('Excluir');
        $action_del->setImage('fa:trash red');
        $action_del->setField('id');

        $this->datagrid->addAction($action_edit);
        $this->datagrid->addAction($action_del);

        $this->datagrid->createModel();

        $panel = new TPanelGroup('Logradouro - Lista');
        $panel->add($this->form);
        $panel->add($this->datagrid);

        parent::add($panel);
    }

    public function onSearch($param)
    {
        $data = $this->form->getData();
        TSession::setValue('search_Logradouro', $data);
        $this->onReload($param);
    }

    public function onReload($param = null)
    {
        try {
            TTransaction::open('comum');

            $repo     = new TRepository('Logradouro');
            $criteria = new TCriteria;

            $data = TSession::getValue('search_Logradouro');

            if ($data) {
                if (!empty($data->id))              { $criteria->add(new TFilter('id', '=', $data->id)); }
                if (!empty($data->nome))            { $criteria->add(new TFilter('nome', 'ilike', "%{$data->nome}%")); }
                if (!empty($data->cep_logradouro))  { $criteria->add(new TFilter('cep_logradouro', 'ilike', "%{$data->cep_logradouro}%")); }
            }

            $objects = $repo->load($criteria);

            // Preload: TipoLogradouro e Bairros (inicial/final)
            $tipoMap   = [];
            $bairroMap = [];

            if ($objects) {
                $idsTipo  = [];
                $idsBairro= [];
                foreach ($objects as $o) {
                    if (!empty($o->id_tipo_logradouro)) { $idsTipo[] = (int)$o->id_tipo_logradouro; }
                    if (!empty($o->id_bairro_inicial))  { $idsBairro[] = (int)$o->id_bairro_inicial; }
                    if (!empty($o->id_bairro_fim))      { $idsBairro[] = (int)$o->id_bairro_fim; }
                }
                if ($idsTipo) {
                    $idsTipo = array_values(array_unique($idsTipo));
                    $repoTipo = new TRepository('TipoLogradouro');
                    $critTipo = new TCriteria;
                    $critTipo->add(new TFilter('id', 'in', $idsTipo));
                    $tipos = $repoTipo->load($critTipo);
                    if ($tipos) {
                        foreach ($tipos as $t) {
                            $tipoMap[(int)$t->id] = $t->nome;
                        }
                    }
                }
                if ($idsBairro) {
                    $idsBairro = array_values(array_unique($idsBairro));
                    $repoBairro = new TRepository('Bairro');
                    $critBairro = new TCriteria;
                    $critBairro->add(new TFilter('id', 'in', $idsBairro));
                    $bairros = $repoBairro->load($critBairro);
                    if ($bairros) {
                        foreach ($bairros as $b) {
                            $bairroMap[(int)$b->id] = $b->nome;
                        }
                    }
                }
            }

            $this->datagrid->clear();
            if ($objects) {
                foreach ($objects as $obj) {
                    $obj->tipo_nome          = $tipoMap[(int)$obj->id_tipo_logradouro] ?? '';
                    $obj->bairro_inicial_nome= $bairroMap[(int)$obj->id_bairro_inicial] ?? '';
                    $obj->bairro_final_nome  = $bairroMap[(int)$obj->id_bairro_fim] ?? '';
                    $this->datagrid->addItem($obj);
                }
            }

            TTransaction::close();
            $this->loaded = true;
        } catch (Exception $e) {
            new TMessage('error', $e->getMessage());
            TTransaction::rollback();
        }
    }

    public function onDelete($param)
    {
        $action = new TAction([$this, 'Delete']);
        $action->setParameters($param);
        new TQuestion('Confirma excluir?', $action);
    }

    public function Delete($param)
    {
        try {
            TTransaction::open('comum');
            $object = new Logradouro($param['id']);
            $object->delete();
            TTransaction::close();
            $this->onReload();
            new TMessage('info', 'Registro excluído');
        } catch (Exception $e) {
            new TMessage('error', $e->getMessage());
            TTransaction::rollback();
        }
    }

    public function show()
    {
        if (!$this->loaded) {
            $this->onReload(func_get_arg(0));
        }
        parent::show();
    }
}
