<?php
use Adianti\Control\TAction;
use Adianti\Control\TPage;
use Adianti\Widget\Container\TPanelGroup;
use Adianti\Widget\Dialog\TMessage;
use Adianti\Widget\Form\TEntry;
use Adianti\Database\TRepository;
use Adianti\Database\TCriteria;
use Adianti\Database\TFilter;
use Adianti\Widget\Wrapper\TQuickForm;
use Adianti\Widget\Datagrid\TDataGridColumn;
use Adianti\Widget\Datagrid\TDataGridAction;
use Adianti\Widget\Datagrid\TDataGrid;
use Adianti\Widget\Dialog\TQuestion;

use Adianti\Database\TTransaction;

// Compat: alias para TSession em diferentes versões
if (!class_exists('TSession')) {
    if (class_exists('\Adianti\Core\TSession')) {
        class_alias('\Adianti\Core\TSession', 'TSession');
    } elseif (class_exists('\Adianti\Registry\TSession')) {
        class_alias('\Adianti\Registry\TSession', 'TSession');
    }
}

class MunicipioLocalidadeList extends TPage
{
    private $form;
    private $datagrid;
    private $loaded;

    public function __construct()
    {
        parent::__construct();

        // Filtro
        $this->form = new TQuickForm('form_search_MunicipioLocalidade');
        $this->form->setFormTitle('Município/Localidade - Busca');

        $id   = new TEntry('id');
        $nome = new TEntry('nome');
        $cep  = new TEntry('cep');

        $this->form->addQuickField('ID',   $id,   100);
        $this->form->addQuickField('Nome', $nome, 300);
        $this->form->addQuickField('CEP',  $cep,  150);

        $this->form->setData( TSession::getValue('search_MunicipioLocalidade') );

        $this->form->addQuickAction('Buscar', new TAction([$this, 'onSearch']), 'fa:search');
        $this->form->addQuickAction('Novo',   new TAction(['MunicipioLocalidadeForm', 'onClear']), 'fa:plus green');

        // Grid
        $this->datagrid = new TDataGrid;
        $this->datagrid->style     = 'width: 100%';
        $this->datagrid->datatable = 'true';

        $col_id          = new TDataGridColumn('id', 'ID', 'center', '8%');
        $col_estado_nome = new TDataGridColumn('estado_nome', 'Estado', 'left', '18%');
        $col_nome        = new TDataGridColumn('nome', 'Nome', 'left', '44%');
        $col_cep         = new TDataGridColumn('cep', 'CEP', 'center', '15%');
        $col_ibge        = new TDataGridColumn('codigo_ibge', 'IBGE', 'center', '15%');

        $this->datagrid->addColumn($col_id);
        $this->datagrid->addColumn($col_estado_nome);
        $this->datagrid->addColumn($col_nome);
        $this->datagrid->addColumn($col_cep);
        $this->datagrid->addColumn($col_ibge);

        // Ações
        $action_edit = new TDataGridAction(['MunicipioLocalidadeForm', 'onEdit']);
        $action_edit->setLabel('Editar');
        $action_edit->setImage('fa:edit blue');
        $action_edit->setField('id');

        $action_del  = new TDataGridAction([$this, 'onDelete']);
        $action_del->setLabel('Excluir');
        $action_del->setImage('fa:trash red');
        $action_del->setField('id');

        $this->datagrid->addAction($action_edit);
        $this->datagrid->addAction($action_del);

        $this->datagrid->createModel();

        $panel = new TPanelGroup('Município/Localidade - Lista');
        $panel->add($this->form);
        $panel->add($this->datagrid);

        parent::add($panel);
    }

    public function onSearch($param)
    {
        $data = $this->form->getData();
        TSession::setValue('search_MunicipioLocalidade', $data);
        $this->onReload($param);
    }

    public function onReload($param = null)
    {
        try {
            TTransaction::open('comum');

            $repository = new TRepository('MunicipioLocalidade');
            $criteria   = new TCriteria;

            $data = TSession::getValue('search_MunicipioLocalidade');

            if ($data) {
                if (!empty($data->id)) {
                    $criteria->add(new TFilter('id', '=', $data->id));
                }
                if (!empty($data->nome)) {
                    $criteria->add(new TFilter('nome', 'ilike', "%{$data->nome}%"));
                }
                if (!empty($data->cep)) {
                    $criteria->add(new TFilter('cep', 'ilike', "%{$data->cep}%"));
                }
            }

            $objects = $repository->load($criteria);

            // Preload dos nomes de Estados (via PDO para evitar "id in $1")
            $estadoMap = [];
            if ($objects) {
                $ids = [];
                foreach ($objects as $o) {
                    if (!empty($o->id_estado)) {
                        $ids[] = (int) $o->id_estado; // sanity
                    }
                }
                $ids = array_values(array_unique($ids));
                if (!empty($ids)) {
                    $in   = implode(',', $ids);
                    $conn = TTransaction::get(); // PDO
                    $sql  = "SELECT id, nome, sigla FROM comum.estado WHERE id IN ($in)";
                    $stm  = $conn->query($sql);
                    $ufs  = $stm ? $stm->fetchAll(\PDO::FETCH_OBJ) : [];
                    foreach ($ufs as $uf) {
                        $estadoMap[(int)$uf->id] = $uf->nome . ' (' . $uf->sigla . ')';
                    }
                }
            }

            $this->datagrid->clear();
            if ($objects) {
                foreach ($objects as $object) {
                    $object->estado_nome = $estadoMap[(int)$object->id_estado] ?? '';
                    $this->datagrid->addItem($object);
                }
            }

            TTransaction::close();
            $this->loaded = true;
        } catch (Exception $e) {
            new TMessage('error', $e->getMessage());
            TTransaction::rollback();
        }
    }

    public function onDelete($param)
    {
        $action = new TAction([$this, 'Delete']);
        $action->setParameters($param);
        new TQuestion('Confirma excluir?', $action);
    }

    public function Delete($param)
    {
        try {
            TTransaction::open('comum');
            $object = new MunicipioLocalidade($param['id']);
            $object->delete();
            TTransaction::close();
            $this->onReload();
            new TMessage('info', 'Registro excluído');
        } catch (Exception $e) {
            new TMessage('error', $e->getMessage());
            TTransaction::rollback();
        }
    }

    public function show()
    {
        if (!$this->loaded) {
            $this->onReload(func_get_arg(0));
        }
        parent::show();
    }
}
