// Exemplo de como manipular a exibição com JavaScript
document.addEventListener('DOMContentLoaded', function () {
    var userGroupId = 1;

    // Controla a exibição de ações baseadas no grupo do usuário
    if (userGroupId === 1) {
        document.getElementById('admin-actions').style.display = 'block';
    } else if (userGroupId === 2) {
        document.getElementById('client-actions').style.display = 'block';
    }

    // Função para verificar se o usuário está logado
    function isLoggedIn() {
        return sessionStorage.setItem("userLoggedIn", "true");
    }
});

// ===== Camadas Base =====
var googleHybrid = L.tileLayer('https://mt1.google.com/vt/lyrs=s&x={x}&y={y}&z={z}', { maxZoom: 22 });
var wmsJequieLayer = createWMSLayer('sisprom:jequie_full_gsd06cm_2025_total');

// ===== Camadas WMS =====
var wmsImoveisLayer = createWMSLayer('sisprom:imoveis_jequie');
var wmsQuadraLayer = createWMSLayer('sisprom:QUADRA_PINDORAMA');
var wmsRuasLayer = createWMSLayer('sisprom:ruas_jequie_2023_wgs84');
var wmsRotuloLoteLayer = createWMSLayer('sisprom:CADASTRO_GERAL_2025_ROTULO');



// ===== Inicializa o Mapa =====
var map = L.map("map", {
    center: [-13.8607, -40.0786],
    zoom: 14,
    minZoom: 10,
    maxZoom: 22,
    zoomControl: false,
    editable: true,
    layers: [wmsJequieLayer]
});

// ===== Adiciona controle de zoom =====
var zoom_bar = new L.Control.ZoomBar({ position: 'topleft' }).addTo(map);

// ===== Adiciona controle de scale =====
L.control.scale({
    metric: true,
    imperial: false,
    maxWidth: 200
}).addTo(map);
map.on('mousemove', updateCoordinates);

// ===== Camada WFS =====
var defaultWFSStyle = { color: '#0000FF', weight: 1, opacity: 0.5, fillOpacity: 0.05 };
var wfsLayer = L.geoJSON(null, { onEachFeature: onEachWFSFeature, style: defaultWFSStyle });
loadWFS();

// ===== Controle de Camadas =====
var controlLayers = L.control.layers(
    { "Google Hybrid": googleHybrid, "Ortophoto Jan2024": wmsJequieLayer },
    { "Mapa Base": wfsLayer, "Ruas/Avenidas": wmsRuasLayer, "Lotes": wmsRotuloLoteLayer, "Quadras": wmsQuadraLayer },
    { collapsed: window.innerWidth < 768 }
).addTo(map);

map.addLayer(wfsLayer);
map.addLayer(wmsRuasLayer);
map.addLayer(wmsRotuloLoteLayer);

// ===== Funções Auxiliares =====
function createWMSLayer(layerName) {
    return L.tileLayer.wms("https://sisprom-br.com.br:8443/geoserver/sisprom/wms", {
        layers: layerName,
        format: 'image/png',
        transparent: true,
        version: '1.1.0',
        srs: 'EPSG:32724',
        minZoom: 10,
        maxZoom: 22
    });
}

function updateCoordinates(e) {
    document.getElementById('map_coords').innerHTML = `
            <strong>Lat:</strong> ${e.latlng.lat.toFixed(4)}
            <strong>Long:</strong> ${e.latlng.lng.toFixed(4)}
            <strong>Zoom:</strong> ${map.getZoom()}`;
}

// ===== wfsLayer =====
// Aguarde o carregamento completo do WFS antes de processar a inscrição
function loadWFS() {
    fetch('https://sisprom-br.com.br:8443/geoserver/sisprom/wfs?service=WFS&version=2.0.0&request=GetFeature&typeNames=sisprom:Jequie_2025_postgis&outputFormat=application/json&srsName=EPSG:4674')
        .then(response => response.json())
        .then(data => {
            wfsLayer.addData(data);
            console.log("Dados WFS carregados.");
            // Após carregar, tente localizar a inscrição, se houver
            if (inscricao) {
                setTimeout(() => highlightInscricao(inscricao), 500);
            }
        })
        .catch(error => console.error('Erro ao carregar WFS:', error));
}

function onEachWFSFeature(feature, layer) {
    layer.bindPopup(` 
            <div style="font-family: Arial, sans-serif; width: 190px; padding: 10px; border-radius: 8px; background: #fff; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                <h4 style="margin: 0 0 6px; font-size: 16px; color: #2c3e50; text-align: center;">Informações do Lote</h4>
                <hr style="margin: 6px 0 10px; border: none; height: 1px; background: #ccc;">
                <table style="width: 100%; font-size: 13px; line-height: 1.3;">
                    <tr><td><strong>Inscrição:</strong></td><td style="text-align: right;">${feature.properties.inscricao}</td></tr>
                    <tr><td><strong>Lote:</strong></td><td style="text-align: right;">${feature.properties.lote}</td></tr>
                    <tr><td><strong>Quadra:</strong></td><td style="text-align: right;">${feature.properties.quadra}</td></tr>
                    <tr><td><strong>Latitude:</strong></td><td style="text-align: right;">${feature.properties.latitude.toFixed(5)}</td></tr>
                    <tr><td><strong>Longitude:</strong></td><td style="text-align: right;">${feature.properties.longitude.toFixed(5)}</td></tr>
                </table>
                <div style="display: flex; justify-content: space-between; gap: 8px; margin-top: 10px;">
                    <a class="btn btn-info" generator="adianti" 
                        href="index.php?class=Consulta_contribuinteFormView&inscricao=${feature.properties.inscricao}&modo=consulta" 
                        style="flex: 1; display: flex; align-items: center; justify-content: center; gap: 4px; background: #2980b9; color: white; padding: 6px 8px; border-radius: 4px; text-decoration: none; font-size: 13px; font-weight: 500;">
                        <i class="fas fa-building"></i> Consultar
                    </a>
                    <a class="btn btn-warning" generator="adianti" 
                        href="index.php?class=AlvaraView&inscricao=${feature.properties.inscricao}&modo=editar" 
                        style="flex: 1; display: flex; align-items: center; justify-content: center; gap: 4px; background: #e67e22; color: white; padding: 6px 8px; border-radius: 4px; text-decoration: none; font-size: 13px; font-weight: 500;">
                        <i class="fas fa-edit"></i> 
                        <div style="text-align: center; line-height: 1.1;"> Alvará
                        </div>
                    </a>
                </div>            
            </div>
        `);

    drawnItems.addLayer(layer);
    layer.on('click', (e) => highlightFeature(layer, e.latlng));
}

// ===== Destaque de Polígono =====
var highlightedLayer = null;
function highlightFeature(layer, latlng) {
    if (highlightedLayer) wfsLayer.resetStyle(highlightedLayer);
    layer.setStyle({ color: '#FF0000', weight: 3, opacity: 0.7, fillOpacity: 0.1 });
    highlightedLayer = layer;
    map.setView(latlng, 19);
}

// ===== Obtém a inscrição =====
// Função para obter parâmetros da URL
function getQueryParam(param) {
    let urlParams = new URLSearchParams(window.location.search);
    return urlParams.get(param);
}
// Aguarde até que o GeoJSON seja completamente carregado
function highlightInscricao(inscricao) {
    let found = false;

    wfsLayer.eachLayer(layer => {
        if (layer.feature && layer.feature.properties.inscricao === inscricao) {
            let bounds = layer.getBounds(); // Obtém os limites do polígono

            if (bounds.isValid()) { // Verifica se os limites são válidos
                found = true;
                let center = bounds.getCenter(); // Obtém o centro do polígono

                map.setView(center, 19); // Define o zoom fixo para 19

                layer.setStyle({
                    fillColor: 'yellow',
                    color: 'red',
                    weight: 3,
                    fillOpacity: 0.7
                });
                // Simula um clique no polígono para abrir as informações
                layer.fire('click');
            } else {
                console.warn("Os limites do polígono são inválidos para a inscrição:", inscricao);
            }
        }
    });

    if (!found) {
        console.warn("Inscrição não encontrada no WFS:", inscricao);
    }
}

// Obtém a inscrição da URL e aguarda o carregamento
var inscricao = getQueryParam("inscricao");

// ===== Medição de Área e Comprimento =====
var isMeasuring = false;
var measureControl = new L.Control.Measure({
    position: "topleft",
    primaryLengthUnit: "meters",
    secondaryLengthUnit: "kilometers",
    primaryAreaUnit: "sqmeters",
    secondaryAreaUnit: "hectares"
});

document.getElementById("MeasureArea").addEventListener("click", function (e) {
    e.preventDefault();
    isMeasuring = !isMeasuring;
    isMeasuring ? map.addControl(measureControl) : map.removeControl(measureControl);
    this.classList.toggle("selected");
});

// ===== Minha Localização =====
(function () {
    let userLocationMarker = null;
    document.getElementById('widgetUserLocation').addEventListener('click', function () {
        toggleUserLocation();
        this.classList.toggle('selected');
    });
    function toggleUserLocation() {
        if (userLocationMarker) {
            map.removeLayer(userLocationMarker);
            userLocationMarker = null;
        } else {
            navigator.geolocation.getCurrentPosition(
                function (position) {
                    const userLatLng = L.latLng(position.coords.latitude, position.coords.longitude);
                    userLocationMarker = L.marker(userLatLng).addTo(map).bindPopup("Você está aqui!").openPopup();
                    map.setView(userLatLng, 15);
                },
                function (error) {
                    alert("Não foi possível acessar sua localização: " + error.message);
                }
            );
        }
    }
})();

// ===== Pesquisar Inscrição e Nome =====
var searchControl = new L.Control.Search({
    layer: wfsLayer,
    zoom: 20,
    collapsed: false, // Mantém visível
    position: 'topleft',
    initial: false,
    hideMarkerOnCollapse: true,
    propertyName: 'inscricao',
    filterData: customPropertySearch,
    moveToLocation: (latlng, title, map) => map.setView(latlng, 20)
}).addTo(map);

// Função para filtrar busca por Inscrição e Nome
function customPropertySearch(text) {
    var filteredData = {};
    var normalizedText = text.trim().toLowerCase();
    wfsLayer.eachLayer(layer => {
        var props = layer.feature.properties;
        var inscricao = props.inscricao?.toString().toLowerCase() || '';
        var name = props.name?.toString().toLowerCase() || '';
        if (inscricao.includes(normalizedText) || name.includes(normalizedText)) {
            filteredData[props.inscricao || props.name] = layer;
        }
    });
    return filteredData;
}

// Mover os campos de busca para a barra superior
document.getElementById('search-control-container').appendChild(document.querySelector('.leaflet-control-search')); // Move a busca por inscrição e nome

// Impedir que o Enter faça o mapa descer
document.querySelectorAll('.leaflet-control input').forEach(input => {
    input.addEventListener('keydown', function (e) {
        if (e.key === 'Enter') {
            e.preventDefault(); // Bloqueia o Enter
        }
    });
});

// ===== Pesquisar Endereço ===== 
var osmGeocoder = new L.Control.Geocoder({
    collapsed: false,
    position: 'topright',
    text: 'Search',
    title: 'Testing',
}).addTo(map);

// Mover o geocoder para a barra superior
var geocoderEl = document.querySelector('.leaflet-control-geocoder');
var searchContainer = document.getElementById('search-control-container-geocoder');
if (geocoderEl && searchContainer) {
    searchContainer.appendChild(geocoderEl);
}

// ===== Desenhar Polígono =====
var styleEditor;
var styleEditorActive = false; // Controle do estado do StyleEditor
// Evento para ativar/desativar o StyleEditor
document.getElementById('widgetStyleEditor').addEventListener('click', function () {
    var button = this; // Referência ao botão
    if (styleEditorActive) {
        // Remove o StyleEditor do mapa
        if (styleEditor) {
            map.removeControl(styleEditor);
            styleEditor = null; // Redefine a variável
        }
        styleEditorActive = false;
        button.src = "voo360/assets/icons3d/palette.svg"; // Ícone para indicar desativado
        button.classList.remove('selected'); // Remove o contorno
    } else {
        // Adiciona o StyleEditor ao mapa
        styleEditor = L.control.styleEditor({
            position: 'topleft',
        }).addTo(map);
        styleEditorActive = true;
        button.src = "voo360/assets/icons3d/palette.svg"; // Ícone para indicar ativado
        button.classList.add('selected'); // Adiciona o contorno
    }
});
// Variável para rastrear o estado da medição
var isMeasuring = false;
// Adiciona o featureGroup para armazenar os polígonos desenhados e marcadores
var drawnItems = new L.FeatureGroup();
map.addLayer(drawnItems);
// Inicializa o controle de desenho, mas sem adicionar ao mapa ainda
var drawControl = new L.Control.Draw({
    draw: {
        polygon: {
            shapeOptions: {
                stroke: true,
                color: 'black',       // Cor da linha do polígono
                fillColor: 'yellow',  // Cor de preenchimento do polígono
                weight: 2             // Largura da borda
            },
            showArea: true  // Mostrar área automaticamente
        },
        polyline: true,
        rectangle: false,
        circle: false,
        marker: true,
        circlemarker: false
    },
    edit: {
        featureGroup: drawnItems,  // Permite editar os polígonos desenhados e marcadores
        remove: true  // Permite deletar os objetos
    }
});
// Função para alternar a ferramenta de medição ao clicar no ícone
document.getElementById('widgetMeasureArea').addEventListener('click', function (e) {
    e.preventDefault();

    if (isMeasuring) {
        // Desativa o modo de desenho ao remover o controle de medição
        map.removeControl(drawControl);
        isMeasuring = false;
        document.getElementById("widgetMeasureArea").src = "voo360/assets/icons3d/pen.png";
        // Remove a classe .selected do ícone
        document.getElementById("widgetMeasureArea").classList.remove("selected");
    } else {
        // Ativa o modo de desenho ao adicionar o controle de medição
        map.addControl(drawControl);
        isMeasuring = true;
        document.getElementById("widgetMeasureArea").src = "voo360/assets/icons3d/pen.png";
        // Adiciona a classe .selected ao ícone
        document.getElementById("widgetMeasureArea").classList.add("selected");
    }
});
// Captura o evento de criação de um polígono ou marcador e adiciona-o ao mapa
map.on('draw:created', function (e) {
    var type = e.layerType,
        layer = e.layer;
    // Adiciona o objeto desenhado (polígono ou marcador) ao featureGroup
    drawnItems.addLayer(layer);
    // Exibe um log no console para ajudar no desenvolvimento
    if (type === 'marker') {
        console.log('Marcador adicionado: ', layer.getLatLng());
    } else if (type === 'polygon') {
        console.log('Polígono adicionado: ', layer);
    }
});
// Captura o evento de edição de objetos
map.on('draw:edited', function (e) {
    var layers = e.layers;
    layers.eachLayer(function (layer) {
        console.log('Objeto editado:', layer);
    });
});
// Captura o evento de exclusão de objetos
map.on('draw:deleted', function (e) {
    var layers = e.layers;
    layers.eachLayer(function (layer) {
        console.log('Objeto deletado:', layer);
    });
});

// ===== Solicitação de Serviço =====
// Variável de controle do botão de solicitação de serviço
var solicitacaoAtiva = false;

// Botão de solicitação de serviço com tooltip
var btnSolicitacao = L.easyButton(
  '<img src="vendor/geojequie/css/images/servico.png" title="Solicitar serviço" style="width:20px; height:20px;">',
  function(btn, map) {
    solicitacaoAtiva = !solicitacaoAtiva;

    var buttonElement = btn.button;
    if (solicitacaoAtiva) {
      buttonElement.classList.add('selected');
      alert("Modo de Solicitação de Serviços ativado!");
    } else {
      buttonElement.classList.remove('selected');
      alert("Modo desativado.");
    }
  }
).addTo(map);


// Manipulação de clique no mapa para adicionar solicitação
map.on('click', function (e) {
    if (!solicitacaoAtiva) return;

    var { lat, lng } = e.latlng;

    var formHtml = `
        <div style="font-family: 'Segoe UI', sans-serif; padding: 16px; max-width: 100%; box-sizing: border-box; margin: 0 auto; background-color: #ffffff; border-radius: 10px; box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);">
            <h3 style="margin-bottom: 16px; color: #2c3e50; font-size: 1.4em; text-align: center;">📌 Solicitação de Serviço</h3>
        
            <label for="tipoServico" style="font-weight: 600; font-size: 1em; margin-bottom: 6px; display: block;">Tipo de serviço:</label>
            <select id="tipoServico" style="width: 100%; padding: 10px; margin-bottom: 14px; border-radius: 6px; border: 1px solid #ccc; font-size: 0.95em; background-color: #fefefe;">
                <option value="Troca de lâmpada">💡 Troca de lâmpada do poste</option>            
                <option value="Poda de árvores">✂️ Poda de árvores</option>
                <option value="Reparo de buracos">🚧 Reparo de buracos na rua</option>
                <option value="Outro">📝 Outro (descrever abaixo)</option>
            </select>
        
            <label for="detalhesServico" style="font-weight: 600; font-size: 1em; margin-bottom: 6px; display: block;">Detalhes:</label>
            <textarea id="detalhesServico" placeholder="Descreva o problema..." 
                style="width: 100%; height: 100px; padding: 10px; border-radius: 6px; border: 1px solid #ccc; font-size: 0.95em; resize: none; background-color: #fefefe;">
            </textarea>
        
            <button onclick="enviarSolicitacao(${lat}, ${lng})" 
                style="width: 100%; padding: 12px; margin-top: 18px; background-color: #007bff; color: white; border: none; border-radius: 6px; cursor: pointer; font-size: 1.05em; font-weight: 600; transition: background-color 0.3s ease;">
                📤 Enviar Solicitação
            </button>
        </div>`;

    L.popup()
        .setLatLng(e.latlng)
        .setContent(formHtml)
        .openOn(map);
});

// Função para enviar a solicitação
function enviarSolicitacao(lat, lng) {
    var tipoServico = document.getElementById('tipoServico').value;
    var detalhes = document.getElementById('detalhesServico').value;
    var username = document.getElementById('username')?.textContent || 'Desconhecido';
    var userGroupId = document.getElementById('user-group-id')?.textContent || '0';

    // Gerar número de protocolo único
    var now = new Date();
    var protocolo = now.getFullYear() +
        ("0" + (now.getMonth() + 1)).slice(-2) +
        ("0" + now.getDate()).slice(-2) + "-" +
        ("0" + now.getHours()).slice(-2) +
        ("0" + now.getMinutes()).slice(-2) +
        ("0" + now.getSeconds()).slice(-2) + "-" +
        Math.floor(1000 + Math.random() * 9000); // Número aleatório de 4 dígitos

    fetch('vendor/geojequie/salvar_solicitacao.php', {
        method: 'POST',
        body: JSON.stringify({ lat, lng, tipoServico, detalhes, username, userGroupId, status_solicitacao: 'Aberto' }),
        headers: { 'Content-Type': 'application/json' }
    })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'sucesso') {
                alert(`Solicitação enviada com sucesso! Número de protocolo: ${data.protocolo}`);

                // Adicionar o marcador no mapa assim que a solicitação for salva com sucesso
                criarMarker({
                    id: data.protocolo, // Usar o protocolo como ID (ou outro identificador único)
                    lat: lat,
                    lng: lng,
                    tipo_servico: tipoServico,
                    detalhes: detalhes,
                    username: username,
                    status_solicitacao: 'Aberto',
                    data_solicitacao: new Date()
                });

            } else {
                alert("Erro ao enviar solicitação: " + data.mensagem);
            }
            // Fechar o popup do mapa
            map.closePopup();
        })
        .catch(error => console.error("Erro ao enviar solicitação:", error));
}

// Definir os ícones personalizados para cada status
var icons = {
    "Aberto": L.icon({
        iconUrl: 'vendor/geojequie/css/images/ponto-de-pin-vermelho.png', // Caminho do ícone
        iconSize: [25, 41],  // Tamanho do ícone
        iconAnchor: [12, 41], // Posição do ponto do ícone no mapa
        popupAnchor: [1, -34] // Ajuste do popup
    }),
    "Em Andamento": L.icon({
        iconUrl: 'vendor/geojequie/css/images/ponto-de-pin-amarelo.png',
        iconSize: [25, 41],
        iconAnchor: [12, 41],
        popupAnchor: [1, -34]
    }),
    "Resolvido": L.icon({
        iconUrl: 'vendor/geojequie/css/images/ponto-de-pin-verde.png',
        iconSize: [25, 41],
        iconAnchor: [12, 41],
        popupAnchor: [1, -34]
    })
};

// Objeto para armazenar os marcadores
var markers = {};

// Função para criar e armazenar o marcador no mapa
function criarMarker(solicitacao) {
    let status = solicitacao.status_solicitacao || "Aberto"; // Define "Aberto" como padrão se o status for indefinido

    if (!icons[status]) {
        console.error(`Erro: Ícone não encontrado para o status "${status}". Usando padrão.`);
        status = "Aberto";
    }

    let marker = L.marker([solicitacao.lat, solicitacao.lng], {
        icon: icons[status]
    }).addTo(map);

    var popupContent = `
                        <div style="font-family: 'Segoe UI', sans-serif; font-size: 13px; max-width: 240px; padding: 8px;">
                            <p style="margin: 4px 0;"><strong>Protocolo:</strong> ${solicitacao.protocolo}</p>
                            <p style="margin: 4px 0;"><strong>Serviço:</strong> ${solicitacao.tipo_servico}</p>
                            <p style="margin: 4px 0;"><strong>Detalhes:</strong> ${solicitacao.detalhes}</p>
                            <p style="margin: 4px 0;"><strong>Data:</strong> ${new Date(solicitacao.data_solicitacao).toLocaleDateString()}</p>
                            <p style="margin: 4px 0;"><strong>Solicitante:</strong> ${solicitacao.username}</p>
                            <p style="margin: 4px 0;"><strong>Status:</strong> 
                                <span style="font-weight: bold; color: ${solicitacao.status_solicitacao === 'Resolvido' ? '#27ae60' :
            solicitacao.status_solicitacao === 'Em Andamento' ? '#f39c12' :
                '#c0392b'
        };">
                                    ${solicitacao.status_solicitacao}
                                </span>
                            </p>
                            <button onclick="deletarSolicitacao('${solicitacao.id}')"
                                style="background-color: #e74c3c; color: white; border: none; padding: 6px 10px; border-radius: 4px; cursor: pointer; font-size: 13px; width: 100%; margin-top: 6px; transition: background-color 0.2s ease;">
                                🗑️ Excluir Solicitação
                            </button>
                        </div>`;

    marker.sollicitacao = solicitacao; // Armazenar os dados no marcador

    // Adicionar botões de admin
    if (userGroupId === "1") {
        popupContent += `
                <div style="margin-top: 10px; text-align: center;">
                    <button onclick="alterarStatus('${solicitacao.id}', 'Em Andamento', ${marker._latlng.lat}, ${marker._latlng.lng})"
                        style="background-color: orange; color: white; border: none; padding: 5px 10px; margin: 5px; border-radius: 5px; cursor: pointer;">
                        Em Andamento
                    </button>
                    <button onclick="alterarStatus('${solicitacao.id}', 'Resolvido', ${marker._latlng.lat}, ${marker._latlng.lng})"
                        style="background-color: green; color: white; border: none; padding: 5px 10px; margin: 5px; border-radius: 5px; cursor: pointer;">
                        Resolvido
                    </button>
                    
                </div>
            `;
    }

    marker.bindPopup(popupContent);
    markers[solicitacao.id] = marker;
}

document.addEventListener("DOMContentLoaded", function () {
    if (userGroupId != 1) {
        document.getElementById("btnExcluir").style.display = "none";
    }
});

// Função para alterar o status da solicitação e trocar o ícone do marcador
function alterarStatus(solicitacaoId, novoStatus) {
    var userGroupId = document.getElementById('user-group-id').textContent;

    fetch('vendor/geojequie/atualizar_status_solicitacao.php', {
        method: 'POST',
        body: JSON.stringify({ id: solicitacaoId, status_solicitacao: novoStatus, userGroupId: userGroupId }),
        headers: { 'Content-Type': 'application/json' }
    })
        .then(response => response.json())
        .then(data => {
            console.log("Resposta do servidor:", data);
            if (data.status === 'sucesso') {
                alert("Status atualizado para: " + novoStatus);

                // Atualizar o ícone do marcador correspondente
                let marker = markers[solicitacaoId];
                if (marker) {
                    marker.setIcon(icons[novoStatus]); // Altera o ícone do marcador
                }
            } else {
                alert("Erro ao atualizar o status: " + data.mensagem);
            }
        })
        .catch(error => console.error("Erro ao atualizar o status:", error));
}

// Carregar solicitações existentes do servidor
var userGroupId = document.getElementById('user-group-id')?.textContent || '0';
var username = document.getElementById('username')?.textContent || 'Desconhecido';

// Função para carregar as solicitações existentes do servidor
fetch('vendor/geojequie/buscar_solicitacoes.php')
    .then(response => response.json())
    .then(data => {
        data.forEach(solicitacao => {
            if (userGroupId === "1" || solicitacao.username === username) {
                criarMarker(solicitacao); // Cria o marcador com o protocolo
            }
        });
    })
    .catch(error => console.error("Erro ao carregar solicitações:", error));

// Função para deletar a solicitação
function deletarSolicitacao(solicitacaoId) {
    if (confirm("Tem certeza que deseja excluir esta solicitação?")) {
        fetch('vendor/geojequie/excluir_solicitacao.php', {
            method: 'POST',
            body: JSON.stringify({ id: solicitacaoId }),
            headers: { 'Content-Type': 'application/json' }
        })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'sucesso') {
                    // Remover o marcador do mapa
                    var marker = markers[solicitacaoId];
                    if (marker) {
                        map.removeLayer(marker); // Remove o marcador do mapa
                        delete markers[solicitacaoId]; // Deleta o marcador do objeto
                    }
                    alert("Solicitação excluída com sucesso!");
                } else {
                    alert("Erro ao excluir solicitação: " + data.mensagem);
                }
            })
            .catch(error => console.error("Erro ao excluir solicitação:", error));
    }
}

// Criar um painel de filtro personalizado apenas para o Grupo 1
if (userGroupId === "1") {
    var filterControl = L.control({ position: 'topright' });

    filterControl.onAdd = function (map) {
        var div = L.DomUtil.create('div', 'info filter-container');
        div.innerHTML = `
                            <div style="font-family: 'Segoe UI', sans-serif; padding: 12px 16px; background-color: #fff; border-radius: 8px; box-shadow: 0 3px 10px rgba(0, 0, 0, 0.1); width: 220px; font-size: 13px;">
                                <h4 style="text-align: center; font-size: 16px; color: #2c3e50; margin-bottom: 12px; font-weight: 600;">🎯 Filtro de Solicitações</h4>                                
                                <label for="filtroStatus" style="font-weight: 500; margin-bottom: 6px; display: block;">Status:</label>
                                <select id="filtroStatus" style="width: 100%; padding: 8px; margin-bottom: 12px; border-radius: 4px; border: 1px solid #ccc; background-color: #fdfdfd;">
                                    <option value="">Todos</option>
                                    <option value="Aberto">Aberto</option>
                                    <option value="Em Andamento">Em Andamento</option>
                                    <option value="Resolvido">Resolvido</option>
                                </select>                    
                                <label for="filtroTipoServico" style="font-weight: 500; margin-bottom: 6px; display: block;">Tipo de Serviço:</label>
                                <select id="filtroTipoServico" style="width: 100%; padding: 8px; margin-bottom: 14px; border-radius: 4px; border: 1px solid #ccc; background-color: #fdfdfd;">
                                    <option value="">Todos</option>
                                    <option value="Troca de lâmpada">💡 Troca de lâmpada</option>
                                    <option value="Poda de árvores">✂️ Poda de árvores</option>
                                    <option value="Reparo de buracos">🚧 Reparo de buracos na rua</option>
                                    <option value="Outro">📝 Outro</option>
                                </select>                    
                                <button onclick="filtrarSolicitacoes()" 
                                    style="width: 100%; padding: 10px; background-color: #007bff; color: #fff; border: none; border-radius: 4px; cursor: pointer; font-size: 13px; font-weight: bold; transition: background-color 0.3s;">
                                    🔍 Filtrar
                                </button>
                            </div>`;
        return div;
    };

    // Adiciona o painel de filtro ao mapa apenas para Grupo 2
    filterControl.addTo(map);
}

// Função para filtrar as solicitações no mapa
function filtrarSolicitacoes() {
    var statusFiltro = document.getElementById('filtroStatus').value;
    var tipoServicoFiltro = document.getElementById('filtroTipoServico').value;

    // Percorrer todos os marcadores e verificar se o status e tipo de serviço correspondem ao filtro
    for (var id in markers) {
        var marker = markers[id];
        var solicitacao = marker.sollicitacao;

        // Exibir ou ocultar o marcador com base no filtro
        var exibir = true;

        // Verificar se o status corresponde ao filtro
        if (statusFiltro && solicitacao.status_solicitacao !== statusFiltro) {
            exibir = false;
        }

        // Verificar se o tipo de serviço corresponde ao filtro
        if (tipoServicoFiltro && solicitacao.tipo_servico !== tipoServicoFiltro) {
            exibir = false;
        }

        // Mostrar ou ocultar o marcador
        if (exibir) {
            marker.addTo(map); // Exibir o marcador
        } else {
            map.removeLayer(marker); // Ocultar o marcador
        }
    }
}

// ===== PrintMap =====
var printerControl = L.easyPrint({
    tileLayer: wmsJequieLayer,
    sizeModes: ['Current', 'A4Landscape', 'A4Portrait'],
    filename: 'mapa_impresso',
    exportOnly: false,
    hideControlContainer: true
}).addTo(map);

function manualPrint() {
    printer.printMap('CurrentSize', 'MyManualPrint')
}

// ===== imagens 360° =====
// Lista de imagens 360°
var fotos360 = [
  { lat: -13.837505, lng: -40.073685, imagem: 'voo360/DJI_0604.jpg', descricao: '360°...'},
  { lat: -13.838240, lng: -40.073859, imagem: 'voo360/DJI_0605.jpg', descricao: '360°...'},
  { lat: -13.838786, lng: -40.074009, imagem: 'voo360/DJI_0606.jpg', descricao: '360°...'},
  { lat: -13.839273, lng: -40.074159, imagem: 'voo360/DJI_0607.jpg', descricao: '360°...'},
  { lat: -13.839824, lng: -40.074302, imagem: 'voo360/DJI_0608.jpg', descricao: '360°...'},
  { lat: -13.840317, lng: -40.074449, imagem: 'voo360/DJI_0609.jpg', descricao: '360°...'},
  { lat: -13.840880, lng: -40.074627, imagem: 'voo360/DJI_0610.jpg', descricao: '360°...'},
  { lat: -13.841671, lng: -40.075071, imagem: 'voo360/DJI_0611.jpg', descricao: '360°...'},
  { lat: -13.841763, lng: -40.074604, imagem: 'voo360/DJI_0612.jpg', descricao: '360°...'},
  { lat: -13.840070, lng: -40.075544, imagem: 'voo360/DJI_0613.jpg', descricao: '360°...'},
  { lat: -13.839637, lng: -40.075401, imagem: 'voo360/DJI_0614.jpg', descricao: '360°...'},
  { lat: -13.839111, lng: -40.075256, imagem: 'voo360/DJI_0615.jpg', descricao: '360°...'},
  { lat: -13.838587, lng: -40.075108, imagem: 'voo360/DJI_0616.jpg', descricao: '360°...'},
  { lat: -13.838043, lng: -40.074950, imagem: 'voo360/DJI_0617.jpg', descricao: '360°...'},
  { lat: -13.837050, lng: -40.074658, imagem: 'voo360/DJI_0618.jpg', descricao: '360°...'},
  { lat: -13.836833, lng: -40.075176, imagem: 'voo360/DJI_0619.jpg', descricao: '360°...'},
  { lat: -13.836621, lng: -40.075670, imagem: 'voo360/DJI_0620.jpg', descricao: '360°...'},
  { lat: -13.837949, lng: -40.075516, imagem: 'voo360/DJI_0621.jpg', descricao: '360°...'},
  { lat: -13.839018, lng: -40.075796, imagem: 'voo360/DJI_0622.jpg', descricao: '360°...'},
  { lat: -13.839965, lng: -40.076059, imagem: 'voo360/DJI_0623.jpg', descricao: '360°...'},
  { lat: -13.840848, lng: -40.075654, imagem: 'voo360/DJI_0624.jpg', descricao: '360°...'},
  { lat: -13.840878, lng: -40.073405, imagem: 'voo360/DJI_0625.jpg', descricao: '360°...'},
  { lat: -13.841502, lng: -40.073466, imagem: 'voo360/DJI_0626.jpg', descricao: '360°...'},
  { lat: -13.840541, lng: -40.073329, imagem: 'voo360/DJI_0627.jpg', descricao: '360°...'},
  { lat: -13.840043, lng: -40.073208, imagem: 'voo360/DJI_0628.jpg', descricao: '360°...'},
  { lat: -13.839483, lng: -40.073033, imagem: 'voo360/DJI_0629.jpg', descricao: '360°...'},
  { lat: -13.838970, lng: -40.072892, imagem: 'voo360/DJI_0630.jpg', descricao: '360°...'},
  { lat: -13.838451, lng: -40.072742, imagem: 'voo360/DJI_0631.jpg', descricao: '360°...'},
  { lat: -13.837926, lng: -40.072590, imagem: 'voo360/DJI_0632.jpg', descricao: '360°...'},
  { lat: -13.838324, lng: -40.073302, imagem: 'voo360/DJI_0633.jpg', descricao: '360°...'},
  { lat: -13.837547, lng: -40.072427, imagem: 'voo360/Golden_06_25/DJI_0301.jpg', descricao: '360°...'},
  { lat: -13.836138, lng: -40.071772, imagem: 'voo360/Golden_06_25/DJI_0302.jpg', descricao: '360°...'},
  { lat: -13.835502, lng: -40.070525, imagem: 'voo360/Golden_06_25/DJI_0303.jpg', descricao: '360°...'},
  { lat: -13.835572, lng: -40.071616, imagem: 'voo360/Golden_06_25/DJI_0304.jpg', descricao: '360°...'},
  { lat: -13.834347, lng: -40.071286, imagem: 'voo360/Golden_06_25/DJI_0305.jpg', descricao: '360°...'},
  { lat: -13.833578, lng: -40.071125, imagem: 'voo360/Golden_06_25/DJI_0306.jpg', descricao: '360°...'},
  { lat: -13.832072, lng: -40.070722, imagem: 'voo360/Golden_06_25/DJI_0307.jpg', descricao: '360°...'},
  { lat: -13.831044, lng: -40.071267, imagem: 'voo360/Golden_06_25/DJI_0308.jpg', descricao: '360°...'},
  { lat: -13.830831, lng: -40.072133, imagem: 'voo360/Golden_06_25/DJI_0309.jpg', descricao: '360°...'},
  { lat: -13.830458, lng: -40.073606, imagem: 'voo360/Golden_06_25/DJI_0310.jpg', descricao: '360°...'},
  { lat: -13.829864, lng: -40.076189, imagem: 'voo360/Golden_06_25/DJI_0311.jpg', descricao: '360°...'},
  { lat: -13.835483, lng: -40.073442, imagem: 'voo360/Golden_06_25/DJI_0312.jpg', descricao: '360°...'},
  { lat: -13.834703, lng: -40.073217, imagem: 'voo360/Golden_06_25/DJI_0313.jpg', descricao: '360°...'},
  { lat: -13.835214, lng: -40.072269, imagem: 'voo360/Golden_06_25/DJI_0314.jpg', descricao: '360°...'},
  { lat: -13.834086, lng: -40.072156, imagem: 'voo360/Golden_06_25/DJI_0315.jpg', descricao: '360°...'},
  { lat: -13.833683, lng: -40.072608, imagem: 'voo360/Golden_06_25/DJI_0316.jpg', descricao: '360°...'},
  { lat: -13.833603, lng: -40.073750, imagem: 'voo360/Golden_06_25/DJI_0317.jpg', descricao: '360°...'},
  { lat: -13.833008, lng: -40.072894, imagem: 'voo360/Golden_06_25/DJI_0318.jpg', descricao: '360°...'},
  { lat: -13.832814, lng: -40.071519, imagem: 'voo360/Golden_06_25/DJI_0319.jpg', descricao: '360°...'},
  { lat: -13.832350, lng: -40.073472, imagem: 'voo360/Golden_06_25/DJI_0320.jpg', descricao: '360°...'},
  { lat: -13.832111, lng: -40.074200, imagem: 'voo360/Golden_06_25/DJI_0321.jpg', descricao: '360°...'},
  { lat: -13.832117, lng: -40.071389, imagem: 'voo360/Golden_06_25/DJI_0322.jpg', descricao: '360°...'},
  { lat: -13.831847, lng: -40.072844, imagem: 'voo360/Golden_06_25/DJI_0323.jpg', descricao: '360°...'},
  { lat: -13.831042, lng: -40.073736, imagem: 'voo360/Golden_06_25/DJI_0324.jpg', descricao: '360°...'},
  { lat: -13.830703, lng: -40.075106, imagem: 'voo360/Golden_06_25/DJI_0325.jpg', descricao: '360°...'},
  { lat: -13.831325, lng: -40.075258, imagem: 'voo360/Golden_06_25/DJI_0326.jpg', descricao: '360°...'},
  { lat: -13.833947, lng: -40.074439, imagem: 'voo360/Golden_06_25/DJI_0327.jpg', descricao: '360°...'},
];

// Definir ícone de drone
var droneIcon = L.icon({
  iconUrl: 'voo360/logo/camera-3605.png',
  iconSize: [32, 32],
  iconAnchor: [16, 32],
  popupAnchor: [0, -32],
  shadowUrl: null
});

// Criar LayerGroup (grupo de camadas) para os marcadores
var fotos360Layer = L.layerGroup();
fotos360.forEach(foto => {
  var marker = L.marker([foto.lat, foto.lng], { icon: droneIcon });
  marker.bindTooltip(foto.descricao);
  marker.on('click', () => abrirViewer(foto.imagem));
  fotos360Layer.addLayer(marker);
});

// Abrir o visualizador
function abrirViewer(imagemUrl) {
  document.getElementById("map").style.display = "none";
  document.getElementById("viewer").style.display = "block";
  document.getElementById("back-button").style.display = "block";
  document.getElementById("viewer").innerHTML = "";

  var viewer = new PhotoSphereViewer({
    panorama: imagemUrl,
    container: document.getElementById('viewer'),
    navbar: true,
    loadingImg: 'geojequie/css/images/GPI-LOGO_2s.gif',
  });
}

// Função global para voltar ao mapa
function backToMap(event) {
  if (event) event.preventDefault();

  document.getElementById("viewer").style.display = "none";
  document.getElementById("map").style.display = "block";
  document.getElementById("back-button").style.display = "none";
  document.getElementById("viewer").innerHTML = "";
}

// Alternar marcadores 360° ao clicar no botão "geo360"
var fotos360Ativado = false;
document.getElementById('geo360').addEventListener('click', () => {
  var botao = document.getElementById('geo360');

  if (!fotos360Ativado) {
    map.addLayer(fotos360Layer);
    botao.classList.add('ativo'); // ativa visual
    fotos360Ativado = true;
  } else {
    map.removeLayer(fotos360Layer);
    botao.classList.remove('ativo'); // desativa visual
    fotos360Ativado = false;
  }
});



