// ==========================
// 📍 MARCADOR DE BUSCA
// ==========================
var markerSource = new ol.source.Vector();
var markerLayer = new ol.layer.Vector({
  source: markerSource,
  style: new ol.style.Style({
    image: new ol.style.Icon({
      src: 'vendor/geojequie/css/images/marcador.png', // ícone marcador
      crossOrigin: 'anonymous',
      anchor: [0.5, 1],
      scale: 0.1,      // Ajuste conforme o tamanho desejado
    })
  })
});

// ====> Camada base ortofoto <====
var baseOrthoLayer = new ol.layer.Tile({
  source: new ol.source.TileWMS({
    url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
    params: {
      LAYERS: 'sisprom:jequie_full_gsd06cm_2025_total',
      TILED: true,
      FORMAT: 'image/jpeg'
    },
    serverType: 'geoserver',
    crossOrigin: 'anonymous'
  }),
  visible: true
});

// ====> Camada base OSM (opcional) <====
var osmLayer = new ol.layer.Tile({
  source: new ol.source.OSM(),
  visible: false
});

// ====> Mapa OL <====
var map = new ol.Map({
  target: 'mapfisco',
  layers: [
    baseOrthoLayer,
    osmLayer,
    markerLayer // 👈 adiciona o layer de marcadores
  ],
  view: new ol.View({
    center: ol.proj.fromLonLat([-40.0799, -13.8509]), // Jequié - BA
    zoom: 13
  })
});

window.map = map; // torna o mapa global

// ==========================
// 🔎 Função para adicionar marcador
// ==========================
function addMarker(lon, lat) {
  markerSource.clear(); // limpa marcador anterior
  var marker = new ol.Feature({
    geometry: new ol.geom.Point(ol.proj.fromLonLat([lon, lat]))
  });
  markerSource.addFeature(marker);

  // Centraliza no marcador
  map.getView().animate({
    center: ol.proj.fromLonLat([lon, lat]),
    zoom: 17,
    duration: 800
  });
}
window.addMarker = addMarker;

// ==========================
// 📋 Painel de busca por coordenada
// ==========================
function openMarkerPanel() {
  if (!setActiveTool('#btn-marker', "Busca por Coordenada")) return;

  document.getElementById("slide-title").textContent = "Busca por Coordenada";
  document.getElementById("slide-content").innerHTML = `
    <div style="font-family: Arial, sans-serif; font-size: 14px; color:#333;">
      <p style="margin-bottom: 12px; font-weight: bold; font-size: 15px; text-align:center;">
        
      </p>

      <div style="display: flex; flex-direction: column; align-items: center; gap: 8px; margin-bottom: 12px;">
        <input id="input-lon" type="number" step="0.00001" placeholder="Longitude"
               style="width:90%; padding:8px 10px; border:1px solid #bbb; border-radius:6px; font-size:13px; outline:none;">
        <input id="input-lat" type="number" step="0.00001" placeholder="Latitude"
               style="width:90%; padding:8px 10px; border:1px solid #bbb; border-radius:6px; font-size:13px; outline:none;">
      </div>

      <section class="measurement-tools">
      <div class="measurement-tools__buttons tool-group">
        <!-- Botão Buscar -->
        <button type="button" onclick="searchMarker()" title="Buscar por coordenada" class="btn-icon" aria-label="Buscar por coordenada">
          <i class="fas fa-search"></i>
        </button>

        <!-- Botão Minha Localização -->
        <button type="button" onclick="locateMe()" title="Minha localização atual" class="btn-icon" aria-label="Minha localização">
          <i class="fas fa-location-dot"></i>
        </button>
      </div>    
    </section>
    </div>
  `;
}
window.openMarkerPanel = openMarkerPanel;


// ==========================
// 🔎 Executa busca no mapa
// ==========================
function searchMarker() {
  const lon = parseFloat(document.getElementById("input-lon").value);
  const lat = parseFloat(document.getElementById("input-lat").value);

  if (isNaN(lon) || isNaN(lat)) {
    alert("Por favor, insira valores válidos de longitude e latitude.");
    return;
  }
  addMarker(lon, lat);
}
window.searchMarker = searchMarker;

function locateMe() {
  if (!navigator.geolocation) {
    alert("Seu navegador não suporta geolocalização.");
    return;
  }

  navigator.geolocation.getCurrentPosition(
    (pos) => {
      const lon = pos.coords.longitude;
      const lat = pos.coords.latitude;
      addMarker(lon, lat);
    },
    (err) => {
      alert("Não foi possível obter sua localização: " + err.message);
    },
    { enableHighAccuracy: true, timeout: 10000 }
  );
}
window.locateMe = locateMe;

// ==========================
// CONTROLES DE ESCALA E COORDENADAS
// ==========================
// Barra de escala
var scaleLine = new ol.control.ScaleLine({
  bar: true,
  steps: 4,
  text: true,
  minWidth: 80,
  className: 'ol-scale-bar'
});
map.addControl(scaleLine);

// Coordenadas
map.on('pointermove', function (evt) {
  if (!evt.dragging) {
    var coord = ol.proj.toLonLat(evt.coordinate);
    var display = document.querySelector('#bottom-panel div:last-child');
    if (display) {
      display.textContent = `Lon: ${coord[0].toFixed(5)}, Lat: ${coord[1].toFixed(5)}`;
    }
  }
  map.getTargetElement().style.cursor =
    map.forEachFeatureAtPixel(evt.pixel, f => f) ? 'pointer' : '';
});

// === INSERIR ESTE BLOCO DEPOIS QUE O MAPA JÁ EXISTIR (window.map) ===
(function () {
  if (!window.map) return;

  // ==========================
  // 🎯 CAMADA DE DESTAQUE (HIGHLIGHT)
  // ==========================
  const highlightSource = new ol.source.Vector();
  const highlightLayer = new ol.layer.Vector({
    source: highlightSource,
    style: new ol.style.Style({
      stroke: new ol.style.Stroke({ color: '#ff0000', width: 3 }),
      fill: new ol.style.Fill({ color: 'rgba(255, 0, 0, 0.1)' })
    })
  });

  // Coleção de feições selecionadas
  const selectedFeatures = new ol.Collection();

  // ==========================
  // 🎨 ESTILOS PADRÕES
  // ==========================
  const defaultStyle = new ol.style.Style({
    stroke: new ol.style.Stroke({ color: '#0057e7', width: 2 }),
    fill: new ol.style.Fill({ color: 'rgba(0, 87, 231, 0.08)' })
  });
  const highlightStyle = new ol.style.Style({
    stroke: new ol.style.Stroke({ color: '#ff6600', width: 3 }),
    fill: new ol.style.Fill({ color: 'rgba(255, 165, 0, 0.2)' })
  });
  const labelBaseStyle = new ol.style.Style({
    text: new ol.style.Text({
      font: '12px Calibri, sans-serif',
      fill: new ol.style.Fill({ color: '#ff00b3ff' }),
      stroke: new ol.style.Stroke({ color: '#fff', width: 3 }),
    })
  });

  // ==========================
  // 🌐 CAMADA WFS (PARCELAMENTO)
  // ==========================
  const wfsSource = new ol.source.Vector({
    format: new ol.format.GeoJSON(),
    url: function (extent) {
      return 'https://sisprom-br.com.br:8443/geoserver/sisprom/wfs?' +
        'service=WFS&version=2.0.0&request=GetFeature&' +
        'typeNames=sisprom:Jequie_2025_postgis&' +
        'outputFormat=application/json&' +
        'srsName=EPSG:3857&' +
        'bbox=' + extent.join(',') + ',EPSG:3857';
    },
    strategy: ol.loadingstrategy.tile(ol.tilegrid.createXYZ({ maxZoom: 18 }))
  });

  const wfsLayer = new ol.layer.Vector({
    source: wfsSource,
    minZoom: 15,
    style: function (feature, resolution) {
      const styles = [];
      const isSelected = selectedFeatures.getArray().includes(feature);
      styles.push(isSelected ? highlightStyle : defaultStyle);

      const zoom = map.getView().getZoomForResolution(resolution);
      if (zoom >= 17) {
        const nome = feature.get('lote') || feature.get('id');
        if (nome) {
          const label = labelBaseStyle.clone();
          label.getText().setText(String(nome));
          styles.push(label);
        }
      }
      return styles;
    }
  });

  // ==========================
  // 📍 MARCADOR DE BUSCA (reaproveita se já existir)
  // ==========================
  if (!window.markerSource || !window.markerLayer) {
    window.markerSource = new ol.source.Vector();
    window.markerLayer = new ol.layer.Vector({
      source: window.markerSource,
      style: new ol.style.Style({
        image: new ol.style.Icon({
          src: 'vendor/geojequie/Mapa_Jequie_Fiscal/img/marker.png', // hospede local para evitar CORS
          crossOrigin: 'anonymous',
          anchor: [0.5, 1],
          scale: 0.05
        })
      })
    });
    map.addLayer(window.markerLayer);
  }

  // ==========================
  // WMS AUXILIARES
  // ==========================
  const imoveisLayer = new ol.layer.Tile({
    source: new ol.source.TileWMS({
      url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
      params: {
        LAYERS: 'sisprom:imoveis_jequie',
        STYLES: 'estilo_por_pavimento',
        TILED: true,
        FORMAT: 'image/png',
        TRANSPARENT: true
      },
      serverType: 'geoserver',
      crossOrigin: 'anonymous'
    }),
    visible: false,
    minZoom: 15
  });

  const ruasJequieLayer = new ol.layer.Tile({
    source: new ol.source.TileWMS({
      url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
      params: {
        LAYERS: 'sisprom:ruas_jequie_2023_wgs84',
        TILED: true,
        FORMAT: 'image/png',
        TRANSPARENT: true
      },
      serverType: 'geoserver',
      crossOrigin: 'anonymous'
    }),
    visible: false,
    zIndex: 25
  });

  const intersecaoLayer = new ol.layer.Tile({
    source: new ol.source.TileWMS({
      url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
      params: {
        LAYERS: 'sisprom:sigef_intersecao',
        TILED: true,
        FORMAT: 'image/png',
        TRANSPARENT: true
      },
      serverType: 'geoserver',
      crossOrigin: 'anonymous'
    }),
    visible: false,
    zIndex: 26
  });

  const perimentroLayer = new ol.layer.Tile({
    source: new ol.source.TileWMS({
      url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
      params: {
        LAYERS: 'sisprom:perimentro_urbano',
        TILED: true,
        FORMAT: 'image/png',
        TRANSPARENT: true
      },
      serverType: 'geoserver',
      crossOrigin: 'anonymous'
    }),
    visible: false,
    zIndex: 26
  });

  // ==========================
  // PAVIMENTOS (WMS com CQL)
  // ==========================
  const pavimentosConfig = ['P-0', 'P-1', 'P-2', 'P-3'];
  const pavimentosLayers = pavimentosConfig.map((pav, idx) => new ol.layer.Tile({
    source: new ol.source.TileWMS({
      url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
      params: {
        LAYERS: 'sisprom:imoveis_jequie',
        TILED: true,
        FORMAT: 'image/png',
        TRANSPARENT: true,
        CQL_FILTER: `pavimentos='${pav}'`
      },
      serverType: 'geoserver',
      crossOrigin: 'anonymous'
    }),
    visible: false,
    zIndex: 22 + idx // ✅ corrigido (antes somava string)
  }));

  // ==========================
  // ADICIONAR CAMADAS AO MAPA
  // ==========================
  map.addLayer(wfsLayer);
  map.addLayer(highlightLayer);
  map.addLayer(imoveisLayer);
  map.addLayer(ruasJequieLayer);
  map.addLayer(intersecaoLayer);
  map.addLayer(perimentroLayer);
  pavimentosLayers.forEach(l => map.addLayer(l));

  // zIndex (ajuste conforme seu stack)
  if (window.baseOrthoLayer) baseOrthoLayer.setZIndex(10);
  if (window.measureLayer)   measureLayer.setZIndex(30);   // sua camada de medição
  wfsLayer.setZIndex(40);
  highlightLayer.setZIndex(50);
  if (window.markerLayer)    markerLayer.setZIndex(60);

  // ==========================
  // SELEÇÃO / HIGHLIGHT NO CLIQUE
  // ==========================
  map.on('singleclick', function (evt) {
    let hit = false;

    map.forEachFeatureAtPixel(
      evt.pixel,
      function (feature, layer) {
        if (layer === wfsLayer) {
          hit = true;

          // sem Shift: limpa seleção anterior
          if (!evt.originalEvent.shiftKey) {
            selectedFeatures.clear();
            highlightSource.clear();
          }

          // adiciona se ainda não estiver selecionada
          if (!selectedFeatures.getArray().includes(feature)) {
            selectedFeatures.push(feature);
          }

          // clona geometria para a camada de highlight
          const clone = new ol.Feature(feature.getGeometry().clone());
          highlightSource.addFeature(clone);
        }
      },
      { layerFilter: l => l === wfsLayer }
    );

    // clicou fora → limpa seleção (se não estiver segurando Shift)
    if (!hit && !evt.originalEvent.shiftKey) {
      selectedFeatures.clear();
      highlightSource.clear();
    }
  });

  // (Opcional) cursor de mão ao passar por features WFS
  map.on('pointermove', function (evt) {
    const hit = map.hasFeatureAtPixel(evt.pixel, { layerFilter: l => l === wfsLayer });
    map.getTargetElement().style.cursor = hit ? 'pointer' : '';
  });

  // ==========================
  // PAINEL: CAMADAS VETORIAIS (usa setActiveTool)
  // ==========================
  function openVectorLayersPanel() {
  if (!setActiveTool('#btn-vlayers', "Camadas Vetoriais")) return;

  document.getElementById("slide-title").textContent = "Camadas Vetoriais";

  let html = `
    <div class="layer-panel">
      <label class="layer-toggle">
        <input type="checkbox" id="toggleWfsLayer" ${wfsLayer.getVisible() ? 'checked' : ''} />
        <span>Parcelamento Urbano (WFS)</span>
      </label>
      <label class="layer-toggle" style="margin-top:10px;">
        <input type="checkbox" id="toggleRuasJequie" ${ruasJequieLayer.getVisible() ? 'checked' : ''} />
        <span>Ruas Jequié 2023</span>
      </label>
      <label class="layer-toggle" style="margin-top:10px;">
        <input type="checkbox" id="toggleIntersecao" ${intersecaoLayer.getVisible() ? 'checked' : ''} />
        <span>Áreas Rurais</span>
      </label>
      <label class="layer-toggle" style="margin-top:10px;">
        <input type="checkbox" id="togglePerimetro" ${perimentroLayer.getVisible() ? 'checked' : ''} />
        <span>Perímetro Urbano</span>
      </label>
  `;

  // 🔒 Apenas grupos 1 e 3 veem os pavimentos
  if (window.userGroupId === 1 || window.userGroupId === 3) {
    html += `
      <hr class="layer-divider" />
      <div class="layer-subtitle">Imóveis por Pavimentos:</div>
      ${pavimentosConfig.map((pav, idx) => `
        <label class="layer-toggle" style="margin-top:6px;">
          <input type="checkbox" id="togglePav${idx}" ${pavimentosLayers[idx].getVisible() ? 'checked' : ''} />
          <span>${pav === 'P-0' ? 'Térreo' : pav.replace('P-', '') + ' Pavimento' + (parseInt(pav.replace('P-', '')) > 1 ? 's' : '')}</span>
        </label>
      `).join('')}
    `;
  }

  html += `</div>`; // fecha layer-panel

  const slideContent = document.getElementById("slide-content");
  slideContent.innerHTML = html;

  document.getElementById("slide-panel").style.display = "block";
  document.querySelectorAll('#tools-panel button').forEach(btn => btn.classList.remove('active'));
  document.querySelector('[onclick="openVectorLayersPanel()"]')?.classList.add('active');

  setTimeout(() => {
    document.getElementById('toggleWfsLayer')?.addEventListener('change', e => wfsLayer.setVisible(e.target.checked));
    document.getElementById('toggleRuasJequie')?.addEventListener('change', e => ruasJequieLayer.setVisible(e.target.checked));
    document.getElementById('toggleIntersecao')?.addEventListener('change', e => intersecaoLayer.setVisible(e.target.checked));
    document.getElementById('togglePerimetro')?.addEventListener('change', e => perimentroLayer.setVisible(e.target.checked));

    // só adiciona listeners dos pavimentos se for grupo 1 ou 3
    if (window.userGroupId === 1 || window.userGroupId === 3) {
      pavimentosConfig.forEach((_, idx) => {
        document.getElementById(`togglePav${idx}`)?.addEventListener('change', e => pavimentosLayers[idx].setVisible(e.target.checked));
      });
    }
  }, 0);
}

  window.openVectorLayersPanel = openVectorLayersPanel;

  // Helper para limpar seleção externamente, se quiser
  function clearHighlights() {
    selectedFeatures.clear();
    highlightSource.clear();
  }
  window.clearHighlights = clearHighlights;

})();
