const WFS_URL = 'https://sisprom-br.com.br:8443/geoserver/sisprom/wfs?service=WFS&version=2.0.0&request=GetFeature&typeNames=sisprom:Jequie_2025_postgis&outputFormat=application/json&srsName=EPSG:3857';

let geojsonFeatures = [];
let olVectorLayer = null;
let olMap = null;
let filtroAtual = { campo: 'setor', valor: null };

function inicializarDashboardSetores() {
  document.getElementById('conteudo-setores').innerHTML = `
    <div class="dashboard-root new-layout">
      <header class="kpi-header">
        <div class="kpi kpi-total">
          <div class="kpi-icon"><span>🏢</span></div>
          <div class="kpi-label">Total Imóveis</div>
          <div class="kpi-value" id="kpi-total">-</div>
        </div>
        <div class="kpi kpi-setores">
          <div class="kpi-icon"><span>📊</span></div>
          <div class="kpi-label">Setores</div>
          <div class="kpi-value" id="kpi-setores">-</div>
        </div>
        <div class="kpi kpi-bairros">
          <div class="kpi-icon"><span>🗺️</span></div>
          <div class="kpi-label">Bairros</div>
          <div class="kpi-value" id="kpi-bairros">-</div>
        </div>
        <div class="kpi kpi-zoneamentos">
          <div class="kpi-icon"><span>🧩</span></div>
          <div class="kpi-label">Zoneamentos</div>
          <div class="kpi-value" id="kpi-zoneamento">-</div>
        </div>
      </header>
      <div class="dashboard-main-area">
        <aside class="sidepanel">
          <h3 class="filtro-title">Filtro</h3>
          <div class="filtro-bloco">
            <label for="filtro-campo">Campo:</label>
            <select id="filtro-campo" class="select-unified">
              <option value="setor">Setor</option>
              <option value="bairro">Bairro</option>
              <option value="zoneamento">Zoneamento</option>
            </select>
          </div>
          <div class="filtro-bloco">
            <label for="filtro-valor">Valor:</label>
            <select id="filtro-valor" class="select-unified"><option value="">Todos</option></select>
          </div>
          <div class="filtro-bloco">
            <label for="busca">Busca:</label>
            <input id="busca" class="search-uni" placeholder="Buscar...">
          </div>
        </aside>
        <main class="dashboard-content">
          <div id="map" class="dashboard-map"></div>
          <div class="dashboard-charts">
            <div class="panel" style="flex:1 1 320px; min-width:220px;">
              <div id="chart-map-bar-morph" style="width:100%;height:260px;"></div>
            </div>
            <div class="panel" style="flex:1 1 220px; min-width:170px;">
              <h4>Imóveis por <span id="label-barra">Bairro</span></h4>
              <div id="chart-bar-bairro" style="width:100%;height:170px;"></div>
            </div>
            <div class="panel" style="flex:1 1 300px; min-width:180px;">
              <h4>TOP 5 Bairros Mais Populosos</h4>
              <div id="chart-top5-bairros" style="width:100%;height:170px;"></div>
            </div>
            <div class="panel" style="flex:1 1 380px; min-width:240px;">
              <h4>Utilização por Setor</h4>
              <div id="chart-utilizacao-por-setor" style="width:100%;height:220px;"></div>
            </div>
          </div>
          <div class="dashboard-chart-tip">
            <span>💡</span>
            Clique em uma barra para visualizar no mapa • Duplo clique para voltar
          </div>
        </main>
      </div>
    </div>
  `;

  carregarDadosWFS().then(features => {
    geojsonFeatures = features;
    inicializarMapa(geojsonFeatures);
    inicializarGraficos(geojsonFeatures);
    configurarFiltros();
    atualizarKPIs(geojsonFeatures);
  });
}

// ========== DADOS ==========
function carregarDadosWFS() {
  return fetch(WFS_URL)
    .then(resp => resp.json())
    .then(geojson => geojson.features || []);
}

// ========== MAPA ==========
function inicializarMapa(features) {
  if (olMap) olMap.setTarget(null);

  const vectorSource = new ol.source.Vector({
    features: new ol.format.GeoJSON().readFeatures(
      { type: "FeatureCollection", features },
      { dataProjection: 'EPSG:3857', featureProjection: 'EPSG:3857' }
    )
  });

  olVectorLayer = new ol.layer.Vector({
    source: vectorSource,
    style: feature => new ol.style.Style({
      stroke: new ol.style.Stroke({ color: '#0088ff', width: 1 }),
      fill: new ol.style.Fill({ color: 'rgba(0, 136, 255, 0.11)' })
    })
  });

  olMap = new ol.Map({
    target: 'map',
    layers: [
      new ol.layer.Tile({ source: new ol.source.OSM() }),
      olVectorLayer
    ],
    view: new ol.View({
      center: ol.extent.getCenter(vectorSource.getExtent()),
      zoom: 13
    })
  });
}

// ========== GRÁFICOS ==========
function inicializarGraficos(features) {
  atualizarGraficos(features);

  document.getElementById('filtro-campo').addEventListener('change', atualizarFiltroCampo);
  document.getElementById('filtro-valor').addEventListener('change', atualizarFiltroValor);
}

function atualizarGraficos(features) {
  // Limpa instâncias ECharts antes de criar novas
  ['chart-bar-bairro', 'chart-map-bar-morph', 'chart-top5-bairros', 'chart-utilizacao-por-setor'].forEach(id => {
    let dom = document.getElementById(id);
    if (dom) {
      const chart = echarts.getInstanceByDom(dom);
      if (chart) chart.dispose();
    }
  });

  // === Gráfico de barras principal ===
  const campo = filtroAtual.campo || 'bairro';
  const dadosBarra = contarPorCampo(features, campo);
  document.getElementById('label-barra').innerText = campo.charAt(0).toUpperCase() + campo.slice(1);

  let chartBar = echarts.init(document.getElementById('chart-bar-bairro'));
  chartBar.setOption({
    title: {
      text: `Imóveis por ${campo.charAt(0).toUpperCase() + campo.slice(1)}`,
      left: 'center',
      top: 16,
      textStyle: { fontWeight: 700, fontSize: 17 }
    },
    grid: { left: 55, right: 22, top: 55, bottom: 40 },
    xAxis: {
      type: 'category',
      data: Object.keys(dadosBarra),
      axisLabel: { fontSize: 15, rotate: 0, margin: 10 }
    },
    yAxis: {
      type: 'value',
      axisLabel: { fontSize: 14, margin: 10 }
    },
    series: [{
      type: 'bar',
      data: Object.values(dadosBarra),
      itemStyle: { color: '#409EFF' },
      label: {
        show: true,
        position: 'top',
        fontSize: 14,
        color: '#215181',
        fontWeight: 600
      }
    }],
    tooltip: { trigger: 'axis' }
  });
  chartBar.off('click');
  chartBar.on('click', params => {
    if (params && params.name) {
      filtroAtual.valor = params.name;
      document.getElementById('filtro-valor').value = params.name;
      aplicarFiltroEAtualizar();
    }
  });

  // === Gráfico de Utilização (horizontal) ===
  const dadosUso = contarPorCampo(features, 'utilizacao') || {};
  const categorias = Object.keys(dadosUso).filter(x => !!x && x !== 'Indefinido');
  const valores = categorias.map(cat => dadosUso[cat]);
  const total = valores.reduce((a, b) => a + b, 0);
  const barrasOrdenadas = categorias.map((c, i) => ({
    name: c,
    value: valores[i]
  })).sort((a, b) => b.value - a.value);

  const chartDom = document.getElementById('chart-map-bar-morph');
  const chart = echarts.init(chartDom);
  chart.clear();

  chart.setOption({
    title: {
      text: 'Distribuição de Utilização',
      left: 'center',
      top: 16,
      textStyle: { fontWeight: 700, fontSize: 18 }
    },
    grid: { left: 130, right: 32, top: 60, bottom: 24 },
    tooltip: {
      trigger: 'item',
      formatter: ({ name, value }) => {
        const perc = total ? ((value / total) * 100).toFixed(1) : 0;
        return `<b>${name}</b><br>Imóveis: <b>${value}</b><br>Percentual: <b>${perc}%</b>`;
      }
    },
    xAxis: {
      type: 'value',
      name: 'Imóveis',
      axisLabel: { fontSize: 14 }
    },
    yAxis: {
      type: 'category',
      data: barrasOrdenadas.map(x => x.name),
      axisLabel: { fontWeight: 600, fontSize: 15, margin: 16 }
    },
    series: [{
      type: 'bar',
      data: barrasOrdenadas.map(x => x.value),
      itemStyle: { color: '#2e6ae6' },
      label: {
        show: true,
        position: 'right',
        fontWeight: 600,
        fontSize: 15,
        color: '#215181',
        formatter: function ({ value }) {
          const perc = total ? ((value / total) * 100).toFixed(1) : 0;
          return `${value} (${perc}%)`;
        }
      }
    }]
  });

  // === TOP 5 bairros ===
  const dadosBairro = contarPorCampo(features, 'bairro');
  const top5Bairros = Object.entries(dadosBairro)
    .filter(x => !!x[0] && x[0] !== 'Indefinido')
    .sort((a, b) => b[1] - a[1])
    .slice(0, 5);

  const chartTop5 = echarts.init(document.getElementById('chart-top5-bairros'));
  chartTop5.setOption({
    title: {
      text: 'TOP 5 Bairros Mais Populosos',
      left: 'center',
      top: 16,
      textStyle: { fontWeight: 700, fontSize: 17 }
    },
    grid: { left: 130, right: 26, top: 60, bottom: 24 },
    xAxis: { type: 'value', name: 'Imóveis', axisLabel: { fontSize: 14 } },
    yAxis: {
      type: 'category',
      data: top5Bairros.map(x => x[0]),
      axisLabel: { fontWeight: 600, fontSize: 15, margin: 14 }
    },
    tooltip: { trigger: 'item', formatter: ({ name, value }) => `<b>${name}</b><br>Imóveis: <b>${value}</b>` },
    series: [{
      type: 'bar',
      data: top5Bairros.map(x => x[1]),
      itemStyle: { color: '#fe9a2e' },
      label: {
        show: true,
        position: 'right',
        fontWeight: 700,
        fontSize: 15,
        color: '#cd7000'
      }
    }]
  });

  // === Barras empilhadas: Utilização por Setor ===
  const setores = [...new Set(features.map(f => f.properties[detectarNomeCampo(features, 'setor')])
    .filter(x => !!x && x !== 'Indefinido'))].sort();
  const utilizacoes = [...new Set(features.map(f => f.properties[detectarNomeCampo(features, 'utilizacao')])
    .filter(x => !!x && x !== 'Indefinido'))].sort();

  const dadosPorSetor = {};
  setores.forEach(setor => {
    dadosPorSetor[setor] = {};
    utilizacoes.forEach(uso => {
      dadosPorSetor[setor][uso] = features.filter(f =>
        f.properties[detectarNomeCampo(features, 'setor')] === setor &&
        f.properties[detectarNomeCampo(features, 'utilizacao')] === uso
      ).length;
    });
  });
  const seriesEmpilhadas = utilizacoes.map(uso => ({
    name: uso,
    type: 'bar',
    stack: 'total',
    emphasis: { focus: 'series' },
    data: setores.map(setor => dadosPorSetor[setor][uso] || 0)
  }));

  let chartStack = echarts.init(document.getElementById('chart-utilizacao-por-setor'));
  chartStack.setOption({
    title: {
      text: 'Utilização por Setor',
      left: 'center',
      top: 30,
      textStyle: { fontWeight: 700, fontSize: 17 }
    },
    legend: {
      data: utilizacoes,
      top: 0,
      left: 0,
      right: 0,
      type: 'scroll',
      orient: 'horizontal',
      itemWidth: 20,
      itemHeight: 12,
      itemGap: 16,
      textStyle: { fontSize: 13, overflow: 'truncate', width: 80 }
    },
    tooltip: { trigger: 'axis' },
    grid: { left: 55, right: 22, top: 60, bottom: 30 },
    xAxis: { type: 'category', data: setores, axisLabel: { fontSize: 14, rotate: 16, margin: 12 } },
    yAxis: { type: 'value', name: 'Imóveis', axisLabel: { fontSize: 13, margin: 8 } },
    series: seriesEmpilhadas
  });
  

  // Clique nas barras do gráfico de utilização
  chart.off('click');
  chart.on('click', params => {
    if (params && params.name) {
      filtroAtual.campo = 'utilizacao';
      filtroAtual.valor = params.name;
      document.getElementById('filtro-campo').value = 'utilizacao';
      popularFiltroValor();
      document.getElementById('filtro-valor').value = params.name;
      aplicarFiltroEAtualizar();
    }
  });

  // Duplo clique reseta filtro
  chartDom.ondblclick = function () {
    filtroAtual.campo = 'setor';
    filtroAtual.valor = null;
    document.getElementById('filtro-campo').value = 'setor';
    popularFiltroValor();
    document.getElementById('filtro-valor').value = '';
    aplicarFiltroEAtualizar();
  };
}

// ========== AUXILIARES ==========
function contarPorCampo(features, campo) {
  const key = detectarNomeCampo(features, campo);
  return features.reduce((agg, feat) => {
    let val = feat.properties[key];
    if (!val) val = 'Indefinido';
    agg[val] = (agg[val] || 0) + 1;
    return agg;
  }, {});
}

function detectarNomeCampo(features, apelido) {
  if (!features.length) return apelido;
  const propNames = Object.keys(features[0].properties);
  const match = propNames.find(name => name.toLowerCase().includes(apelido));
  return match || apelido;
}

// ========== FILTRO E INTERAÇÃO ==========
function configurarFiltros() {
  popularFiltroValor();
  document.getElementById('busca').addEventListener('input', aplicarFiltroEAtualizar);
}

function atualizarFiltroCampo() {
  filtroAtual.campo = document.getElementById('filtro-campo').value;
  filtroAtual.valor = null;
  popularFiltroValor();
  aplicarFiltroEAtualizar();
}

function atualizarFiltroValor() {
  filtroAtual.valor = document.getElementById('filtro-valor').value || null;
  aplicarFiltroEAtualizar();
}

function popularFiltroValor() {
  const campo = filtroAtual.campo;
  const select = document.getElementById('filtro-valor');
  select.innerHTML = `<option value="">Todos</option>`;
  const valoresUnicos = Array.from(new Set(
    geojsonFeatures.map(f => f.properties[detectarNomeCampo(geojsonFeatures, campo)])
      .filter(x => !!x && x !== 'Indefinido')
  ));
  valoresUnicos.sort();
  valoresUnicos.forEach(valor => {
    select.innerHTML += `<option value="${valor}">${valor}</option>`;
  });
}

function aplicarFiltroEAtualizar() {
  let filtrados = geojsonFeatures;
  if (filtroAtual.valor) {
    const key = detectarNomeCampo(geojsonFeatures, filtroAtual.campo);
    filtrados = filtrados.filter(f => f.properties[key] == filtroAtual.valor);
  }
  const busca = document.getElementById('busca').value.trim().toLowerCase();
  if (busca.length > 0) {
    filtrados = filtrados.filter(f => {
      return Object.values(f.properties)
        .some(val => val && String(val).toLowerCase().includes(busca));
    });
  }

  // Atualiza camada do mapa
  const vectorSource = new ol.source.Vector({
    features: new ol.format.GeoJSON().readFeatures(
      { type: "FeatureCollection", features: filtrados },
      { dataProjection: 'EPSG:3857', featureProjection: 'EPSG:3857' }
    )
  });
  olVectorLayer.setSource(vectorSource);

  if (filtrados.length > 0) {
    olMap.getView().fit(vectorSource.getExtent(), { maxZoom: 16, duration: 400 });
  }

  atualizarGraficos(filtrados);
  atualizarKPIs(filtrados);
}

// ========== KPIs ==========
function atualizarKPIs(features) {
  document.getElementById('kpi-total').innerText = features.length;
  document.getElementById('kpi-setores').innerText = new Set(
    features.map(f => f.properties[detectarNomeCampo(features, 'setor')]).filter(x => !!x && x !== 'Indefinido')
  ).size;
  document.getElementById('kpi-bairros').innerText = new Set(
    features.map(f => f.properties[detectarNomeCampo(features, 'bairro')]).filter(x => !!x && x !== 'Indefinido')
  ).size;
  document.getElementById('kpi-zoneamento').innerText = new Set(
    features.map(f => f.properties[detectarNomeCampo(features, 'zoneamento')]).filter(x => !!x && x !== 'Indefinido')
  ).size;
}

// Exporta global
window.inicializarDashboardSetores = inicializarDashboardSetores;
