// Exemplo de como manipular a exibição com JavaScript
document.addEventListener('DOMContentLoaded', function () {
    var userGroupId = 1;

    // Controla a exibição de ações baseadas no grupo do usuário
    if (userGroupId === 1) {
        document.getElementById('admin-actions').style.display = 'block';
    } else if (userGroupId === 2) {
        document.getElementById('client-actions').style.display = 'block';
    }

    // Função para verificar se o usuário está logado
    function isLoggedIn() {
        return sessionStorage.setItem("userLoggedIn", "true");
    }
});

// ===== Camadas Base =====
var googleHybrid = L.tileLayer('https://mt1.google.com/vt/lyrs=s&x={x}&y={y}&z={z}', { maxZoom: 22 });
var wmsJequieLayer = createWMSLayer('sisprom:jequie_full_gsd06cm_2025_total');

// ===== Camadas WMS =====
var wmsImoveisLayer = createWMSLayer('sisprom:imoveis_jequie');
var wmsQuadraLayer = createWMSLayer('sisprom:QUADRA_PINDORAMA');
var wmsRuasLayer = createWMSLayer('sisprom:ruas_jequie_2023_wgs84');
var wmsRotuloLoteLayer = createWMSLayer('sisprom:CADASTRO_GERAL_2025_ROTULO');



// ===== Inicializa o Mapa =====
var map = L.map("map", {
    center: [-13.8607, -40.0786],
    zoom: 14,
    minZoom: 10,
    maxZoom: 22,
    zoomControl: false,
    editable: true,
    layers: [wmsJequieLayer]
});

// ===== Adiciona controle de zoom =====
var zoom_bar = new L.Control.ZoomBar({ position: 'topleft' }).addTo(map);

// ===== Adiciona controle de scale =====
L.control.scale({
    metric: true,
    imperial: false,
    maxWidth: 200
}).addTo(map);
map.on('mousemove', updateCoordinates);

// ===== Camada WFS =====
var defaultWFSStyle = { color: '#0000FF', weight: 1, opacity: 0.5, fillOpacity: 0.05 };
var wfsLayer = L.geoJSON(null, { onEachFeature: onEachWFSFeature, style: defaultWFSStyle });
loadWFS();

// ===== Controle de Camadas =====
var controlLayers = L.control.layers(
    { "Google Hybrid": googleHybrid, "Ortophoto Jan2024": wmsJequieLayer },
    { "Mapa Base": wfsLayer, "Ruas/Avenidas": wmsRuasLayer, "Lotes": wmsRotuloLoteLayer, "Quadras": wmsQuadraLayer },
    { collapsed: window.innerWidth < 768 }
).addTo(map);

map.addLayer(wfsLayer);
map.addLayer(wmsRuasLayer);
map.addLayer(wmsRotuloLoteLayer);

// ===== Funções Auxiliares =====
function createWMSLayer(layerName) {
    return L.tileLayer.wms("https://sisprom-br.com.br:8443/geoserver/sisprom/wms", {
        layers: layerName,
        format: 'image/png',
        transparent: true,
        version: '1.1.0',
        srs: 'EPSG:32724',
        minZoom: 10,
        maxZoom: 22
    });
}

function updateCoordinates(e) {
    document.getElementById('map_coords').innerHTML = `
            <strong>Lat:</strong> ${e.latlng.lat.toFixed(4)}
            <strong>Long:</strong> ${e.latlng.lng.toFixed(4)}
            <strong>Zoom:</strong> ${map.getZoom()}`;
}

// ===== wfsLayer =====
// Aguarde o carregamento completo do WFS antes de processar a inscrição
function loadWFS() {
    fetch('https://sisprom-br.com.br:8443/geoserver/sisprom/wfs?service=WFS&version=2.0.0&request=GetFeature&typeNames=sisprom:Jequie_2025_postgis&outputFormat=application/json&srsName=EPSG:4674')
        .then(response => response.json())
        .then(data => {
            wfsLayer.addData(data);
            console.log("Dados WFS carregados.");
            // Após carregar, tente localizar a inscrição, se houver
            if (inscricao) {
                setTimeout(() => highlightInscricao(inscricao), 500);
            }
        })
        .catch(error => console.error('Erro ao carregar WFS:', error));
}

function onEachWFSFeature(feature, layer) {
    layer.bindPopup(` 
            <div style="font-family: Arial, sans-serif; width: 250px; padding: 10px; border-radius: 8px; background: #fff; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                <h4 style="margin: 0 0 6px; font-size: 16px; color: #2c3e50; text-align: center;">Informações do Lote</h4>
                <hr style="margin: 6px 0 10px; border: none; height: 1px; background: #ccc;">
                <table style="width: 100%; font-size: 13px; line-height: 1.3;">
                    <tr><td><strong>Inscrição:</strong></td><td style="text-align: right;">${feature.properties.inscricao}</td></tr>
                    <tr><td><strong>Lote:</strong></td><td style="text-align: right;">${feature.properties.lote}</td></tr>
                    <tr><td><strong>Quadra:</strong></td><td style="text-align: right;">${feature.properties.quadra}</td></tr>
                    <tr><td><strong>Zoneamento:</strong></td><td style="text-align: right;">${feature.properties.zoneamento}</td></tr>
                    <tr><td><strong>Latitude:</strong></td><td style="text-align: right;">${feature.properties.latitude.toFixed(5)}</td></tr>
                    <tr><td><strong>Longitude:</strong></td><td style="text-align: right;">${feature.properties.longitude.toFixed(5)}</td></tr>
                </table>
                <div style="display: flex; justify-content: space-between; gap: 8px; margin-top: 10px;">
                    <a class="btn btn-info" generator="adianti" 
                        href="index.php?class=Consulta_viabilidadeFormView&inscricao=${feature.properties.inscricao}&modo=consulta" 
                        style="flex: 1; display: flex; align-items: center; justify-content: center; gap: 4px; background: #2980b9; color: white; padding: 6px 8px; border-radius: 4px; text-decoration: none; font-size: 13px; font-weight: 500;">
                        <i class="fas fa-building"></i> Consultar
                    </a>
                </div>            
            </div>
        `);

    
    layer.on('click', (e) => highlightFeature(layer, e.latlng));
}

function setMapPoint(lat, lng) {
    if (window.map) {
        const point = L.latLng(lat, lng);

        // Remove marcador anterior (opcional)
        if (window.viabilidadeMarker) {
            window.map.removeLayer(window.viabilidadeMarker);
        }

        window.viabilidadeMarker = L.marker(point, { title: "Localização do Lote" }).addTo(map)
            .bindPopup(`<b>Localização selecionada</b><br>Lat: ${lat.toFixed(5)}<br>Lng: ${lng.toFixed(5)}`)
            .openPopup();

        map.setView(point, 18);
    } else {
        console.error('Mapa não carregado ainda');
    }
}

// ===== Destaque de Polígono =====
var highlightedLayer = null;
function highlightFeature(layer, latlng) {
    if (highlightedLayer) wfsLayer.resetStyle(highlightedLayer);
    layer.setStyle({ color: '#FF0000', weight: 3, opacity: 0.7, fillOpacity: 0.1 });
    highlightedLayer = layer;
    map.setView(latlng, 19);
}

// ===== Obtém a inscrição =====
// Função para obter parâmetros da URL
function getQueryParam(param) {
    let urlParams = new URLSearchParams(window.location.search);
    return urlParams.get(param);
}
// Aguarde até que o GeoJSON seja completamente carregado
function highlightInscricao(inscricao) {
    let found = false;

    wfsLayer.eachLayer(layer => {
        if (layer.feature && layer.feature.properties.inscricao === inscricao) {
            let bounds = layer.getBounds(); // Obtém os limites do polígono

            if (bounds.isValid()) { // Verifica se os limites são válidos
                found = true;
                let center = bounds.getCenter(); // Obtém o centro do polígono

                map.setView(center, 19); // Define o zoom fixo para 19

                layer.setStyle({
                    fillColor: 'yellow',
                    color: 'red',
                    weight: 3,
                    fillOpacity: 0.7
                });
                // Simula um clique no polígono para abrir as informações
                layer.fire('click');
            } else {
                console.warn("Os limites do polígono são inválidos para a inscrição:", inscricao);
            }
        }
    });

    if (!found) {
        console.warn("Inscrição não encontrada no WFS:", inscricao);
    }
}

// Obtém a inscrição da URL e aguarda o carregamento
var inscricao = getQueryParam("inscricao");