<?php
/**
 * sync_inscricao.php  (PG 9.2 friendly)
 * Consulta a API de inscrição (GET via cURL) e persiste dados em múltiplas tabelas do schema banco_el.
 *
 * Tabelas esperadas (colunas mínimas):
 *  - banco_el.banco_el_inscricao(inscricao_municipal PK, cep, tipo, codigo, numero, memorial, englobado, complemento,
 *      data_inclusao, data_exclusao, data_alteracao, exclusao_logica, processo_inclusao, processo_exclusao, correspondencia_local)
 *  - banco_el.banco_el_pessoa(inscricao_municipal PK, id, nome, tipo, fantasia, documento, inscricao_estadual)
 *  - banco_el.banco_el_logradouro(inscricao_municipal PK, nome, tipo_id, tipo_sigla, tipo_descricao, codigo)
 *  - banco_el.banco_el_bairro(inscricao_municipal PK, nome, codigo)
 *  - banco_el.banco_el_cidade(inscricao_municipal PK, nome, codigo_ibge, uf_sigla, uf_descricao)
 *  - banco_el.banco_el_boletim(id SERIAL PK, inscricao_municipal, campo, descricao, resposta_id, resposta_descricao)
 *
 * Uso:
 *   /vendor/geojequie/sync_inscricao.php?inscricao=01043630188001
 */

declare(strict_types=1);
date_default_timezone_set('America/Bahia');
header('Content-Type: application/json; charset=utf-8');

// ============================
// CONFIG
// ============================
$PG_DSN  = getenv('PG_DSN')  ?: 'pgsql:host=162.240.148.136;port=5432;dbname=wwsisp_tributogis';
$PG_USER = getenv('PG_USER') ?: 'wwsisp_sisprom';
$PG_PASS = getenv('PG_PASS') ?: 'R312@dm1nj3q';

// Endpoint da API que já funciona no seu ambiente online
$API_URL = getenv('API_EL_URL') ?: 'https://geo-el02.cloud.sisprom-br.com.br/vendor/geojequie/consulta_inscricao_el.php';

// Entrada
$inscricao = $_GET['inscricao'] ?? ($_POST['inscricao'] ?? null);
$debug = isset($_GET['debug']) && $_GET['debug'] == '1';

if (!$inscricao) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Informe a inscrição via ?inscricao=...']);
    exit;
}

// ============================
// HELPERS
// ============================
function to_pg_ts($s) {
    if ($s === '' || $s === null) return null;
    $t = strtotime((string)$s);
    return $t ? date('Y-m-d H:i:s', $t) : null;
}
function get_n(array $arr, array $path) {
    $cur = $arr;
    foreach ($path as $k) {
        if (!is_array($cur) || !array_key_exists($k, $cur)) return null;
        $cur = $cur[$k];
    }
    return $cur;
}
function curl_get_json(string $url, int $timeout = 20): array {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_SSL_VERIFYPEER => false, // homologação; em produção ideal é TRUE com CA válido
        CURLOPT_HTTPHEADER => ['Accept: application/json; charset=utf-8'],
    ]);
    $body = curl_exec($ch);
    $errno = curl_errno($ch);
    $err   = curl_error($ch);
    $code  = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($errno) {
        throw new RuntimeException("Erro cURL ao consultar API: {$err}");
    }
    if ($code !== 200) {
        $preview = is_string($body) ? substr($body, 0, 400) : '';
        throw new RuntimeException("API respondeu HTTP {$code}. URL: {$url}. Corpo (início): {$preview}");
    }
    $data = json_decode((string)$body, true);
    if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
        throw new RuntimeException('JSON inválido da API: ' . json_last_error_msg());
    }
    return $data;
}

// ============================
// 1) Consulta API (GET)
// ============================
try {
    $url = $API_URL . '?operacao=consulta&inscricao=' . urlencode($inscricao);
    $payload = curl_get_json($url);
    if ($debug) {
        echo json_encode(['status' => 'debug', 'api_url' => $url, 'keys' => array_keys((array)$payload)], JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT);
        exit;
    }
} catch (Throwable $e) {
    http_response_code(502);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    exit;
}

// Determina a inscrição efetiva (preferência pela API)
$insc = get_n($payload, ['imobiliario', 'inscricaoCadastral']);
if (!$insc) $insc = $inscricao;

// ============================
// 2) Persistência no Postgres (transação)
// ============================
try {
    $pdo = new PDO($PG_DSN, $PG_USER, $PG_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);
    $pdo->beginTransaction();

    // -------- INSCRICAO (tabela principal) --------
    // *IMPORTANTE*: DELETE e INSERT em PREPARES separados (evita "cannot insert multiple commands into a prepared statement")
    $stmtDel = $pdo->prepare("DELETE FROM banco_el.banco_el_inscricao WHERE inscricao_municipal = :insc");
    $stmtDel->execute([':insc' => $insc]);

    $stmtIns = $pdo->prepare("
        INSERT INTO banco_el.banco_el_inscricao (
            inscricao_municipal, cep, tipo, codigo, numero, memorial, englobado,
            complemento, data_inclusao, data_exclusao, data_alteracao,
            exclusao_logica, processo_inclusao, processo_exclusao, correspondencia_local
        ) VALUES (
            :insc, :cep, :tipo, :codigo, :numero, :memorial, :englobado,
            :complemento, :data_inclusao, :data_exclusao, :data_alteracao,
            :exclusao_logica, :processo_inclusao, :processo_exclusao, :correspondencia_local
        )
    ");
    $stmtIns->execute([
        ':insc'                => $insc,
        ':cep'                 => $payload['cep'] ?? null,
        ':tipo'                => $payload['tipo'] ?? null,
        ':codigo'              => isset($payload['codigo']) ? (int)$payload['codigo'] : null,
        ':numero'              => $payload['numero'] ?? null,
        ':memorial'            => $payload['memorial'] ?? null,
        ':englobado'           => $payload['englobado'] ?? null,
        ':complemento'         => $payload['complemento'] ?? null,
        ':data_inclusao'       => to_pg_ts($payload['dataInclusao'] ?? null),
        ':data_exclusao'       => to_pg_ts($payload['dataExclusao'] ?? null),
        ':data_alteracao'      => to_pg_ts($payload['dataAlteracao'] ?? null),
        ':exclusao_logica'     => $payload['exclusaoLogica'] ?? null,
        ':processo_inclusao'   => $payload['processoInclusao'] ?? null,
        ':processo_exclusao'   => $payload['processoExclusao'] ?? null,
        ':correspondencia_local' => $payload['correspondenciaLocal'] ?? null,
    ]);

    // -------- PESSOA (1:1) --------
    $pdo->prepare("DELETE FROM banco_el.banco_el_pessoa WHERE inscricao_municipal = :insc")
        ->execute([':insc' => $insc]);
    $pessoa = $payload['pessoa'] ?? null;
    if (is_array($pessoa)) {
        $stmt = $pdo->prepare("
            INSERT INTO banco_el.banco_el_pessoa (
                inscricao_municipal, id, nome, tipo, fantasia, documento, inscricao_estadual
            ) VALUES (
                :insc, :id, :nome, :tipo, :fantasia, :doc, :ie
            )
        ");
        $stmt->execute([
            ':insc'     => $insc,
            ':id'       => $pessoa['id'] ?? null,
            ':nome'     => $pessoa['nome'] ?? null,
            ':tipo'     => $pessoa['tipo'] ?? null,
            ':fantasia' => $pessoa['fantasia'] ?? null,
            ':doc'      => $pessoa['documento'] ?? null,
            ':ie'       => $pessoa['inscricaoEstadual'] ?? null,
        ]);
    }

    // -------- LOGRADOURO (1:1) --------
    $pdo->prepare("DELETE FROM banco_el.banco_el_logradouro WHERE inscricao_municipal = :insc")
        ->execute([':insc' => $insc]);
    $logr = $payload['logradouro'] ?? null;
    $tipo = is_array($logr) ? ($logr['tipo'] ?? null) : null;
    if (is_array($logr)) {
        $stmt = $pdo->prepare("
            INSERT INTO banco_el.banco_el_logradouro (
                inscricao_municipal, nome, tipo_id, tipo_sigla, tipo_descricao, codigo
            ) VALUES (
                :insc, :nome, :tipo_id, :sigla, :descricao, :codigo
            )
        ");
        $stmt->execute([
            ':insc'      => $insc,
            ':nome'      => $logr['nome'] ?? null,
            ':tipo_id'   => is_array($tipo) ? ($tipo['id'] ?? null) : null,
            ':sigla'     => is_array($tipo) ? ($tipo['sigla'] ?? null) : null,
            ':descricao' => is_array($tipo) ? ($tipo['descricao'] ?? null) : null,
            ':codigo'    => $logr['codigo'] ?? null,
        ]);
    }

    // -------- BAIRRO (1:1) --------
    $pdo->prepare("DELETE FROM banco_el.banco_el_bairro WHERE inscricao_municipal = :insc")
        ->execute([':insc' => $insc]);
    $bairro = $payload['bairro'] ?? null;
    if (is_array($bairro)) {
        $stmt = $pdo->prepare("
            INSERT INTO banco_el.banco_el_bairro (
                inscricao_municipal, nome, codigo
            ) VALUES (
                :insc, :nome, :codigo
            )
        ");
        $stmt->execute([
            ':insc'   => $insc,
            ':nome'   => $bairro['nome'] ?? null,
            ':codigo' => $bairro['codigo'] ?? null,
        ]);
    }

    // -------- CIDADE (1:1) --------
    $pdo->prepare("DELETE FROM banco_el.banco_el_cidade WHERE inscricao_municipal = :insc")
        ->execute([':insc' => $insc]);
    $cidade = $payload['cidade'] ?? null;
    $uf     = is_array($cidade) ? ($cidade['tipo'] ?? null) : null;
    if (is_array($cidade)) {
        $stmt = $pdo->prepare("
            INSERT INTO banco_el.banco_el_cidade (
                inscricao_municipal, nome, codigo_ibge, uf_sigla, uf_descricao
            ) VALUES (
                :insc, :nome, :ibge, :uf, :uf_desc
            )
        ");
        $stmt->execute([
            ':insc'    => $insc,
            ':nome'    => $cidade['nome'] ?? null,
            ':ibge'    => $cidade['codigoIbge'] ?? null,
            ':uf'      => is_array($uf) ? ($uf['sigla'] ?? null) : null,
            ':uf_desc' => is_array($uf) ? ($uf['descricao'] ?? null) : null,
        ]);
    }

    // -------- BOLETIM (N:1) --------
    $pdo->prepare("DELETE FROM banco_el.banco_el_boletim WHERE inscricao_municipal = :insc")
        ->execute([':insc' => $insc]);
    $boletins = get_n($payload, ['imobiliario', 'boletim']);
    if (is_array($boletins)) {
        $stmt = $pdo->prepare("
            INSERT INTO banco_el.banco_el_boletim (
                inscricao_municipal, campo, descricao, resposta_id, resposta_descricao
            ) VALUES (
                :insc, :campo, :descricao, :respid, :respdesc
            )
        ");
        foreach ($boletins as $b) {
            $resp = is_array($b) ? ($b['resposta'] ?? null) : null;
            $stmt->execute([
                ':insc'     => $insc,
                ':campo'    => is_array($b) ? ($b['campo'] ?? null) : null,
                ':descricao'=> is_array($b) ? ($b['descricao'] ?? null) : null,
                ':respid'   => is_array($resp) ? ($resp['id'] ?? null) : null,
                ':respdesc' => is_array($resp) ? ($resp['descricao'] ?? null) : null,
            ]);
        }
    }

    $pdo->commit();
    echo json_encode(['status' => 'ok', 'message' => 'Dados sincronizados com sucesso.', 'inscricao' => $insc], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
    if (isset($pdo) && $pdo->inTransaction()) { $pdo->rollBack(); }
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()], JSON_UNESCAPED_UNICODE);
}
