<?php

declare(strict_types=1);
date_default_timezone_set('America/Bahia');
header('Content-Type: application/json; charset=utf-8');

// ============================
// CONFIG
// ============================
$PG_DSN  = getenv('PG_DSN')  ?: 'pgsql:host=162.240.148.136;port=5432;dbname=wwsisp_tributogis';
$PG_USER = getenv('PG_USER') ?: 'wwsisp_sisprom';
$PG_PASS = getenv('PG_PASS') ?: 'R312@dm1nj3q';
$API_URL = getenv('API_EL_URL') ?: 'https://geo-el02.cloud.sisprom-br.com.br/vendor/geojequie/consulta_inscricao_el.php';
$DELAY_MS = (int)(getenv('BATCH_DELAY_MS') ?: 0);

// ============================
// FUNÇÕES AUXILIARES
// ============================
function to_pg_ts($s) {
    if ($s === '' || $s === null) return null;
    $t = strtotime((string)$s);
    return $t ? date('Y-m-d H:i:s', $t) : null;
}
function get_n(array $arr, array $path) {
    $cur = $arr;
    foreach ($path as $k) {
        if (!is_array($cur) || !array_key_exists($k, $cur)) return null;
        $cur = $cur[$k];
    }
    return $cur;
}
function curl_get_json(string $url, int $timeout = 25): array {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_SSL_VERIFYPEER => false, // ajuste p/ produção
        CURLOPT_HTTPHEADER => ['Accept: application/json; charset=utf-8'],
    ]);
    $body = curl_exec($ch);
    $errno = curl_errno($ch);
    $err   = curl_error($ch);
    $code  = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($errno) throw new RuntimeException("Erro cURL: {$err}");
    if ($code !== 200) {
        $preview = is_string($body) ? substr($body, 0, 400) : '';
        throw new RuntimeException("HTTP {$code}. Corpo(início): {$preview}");
    }
    $data = json_decode((string)$body, true);
    if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
        throw new RuntimeException('JSON inválido: ' . json_last_error_msg());
    }
    return $data;
}

/** Lê inscrições de TXT/CSV (coluna 'inscricao') ou da query 'list' */
function load_inscricoes(?string $path, ?string $listParam): array {
    $vals = [];

    // 1) Lista inline tem prioridade
    if ($listParam) {
        foreach (explode(',', $listParam) as $s) {
            $s = trim($s);
            if ($s !== '') $vals[] = $s;
        }
        return array_values(array_unique($vals));
    }

    // 2) Sem arquivo? nada a fazer
    if (!$path) return [];

    // 3) Se for URL http(s), baixa e lê
    if (preg_match('~^https?://~i', $path)) {
        // Tente via cURL (mais robusto)
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $path,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 25,
            CURLOPT_SSL_VERIFYPEER => false, // ajuste para TRUE se tiver CA válido
        ]);
        $body = curl_exec($ch);
        $errno = curl_errno($ch);
        $err  = curl_error($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($errno) throw new RuntimeException("Falha ao baixar lista: {$err}");
        if ($code !== 200 || !is_string($body)) {
            throw new RuntimeException("Falha ao baixar lista (HTTP {$code}).");
        }
        // Trata como TXT: uma inscrição por linha
        foreach (preg_split("/\r\n|\n|\r/", $body) as $ln) {
            $ln = trim($ln);
            if ($ln === '' || strpos($ln, '#') === 0 || strpos($ln, '//') === 0) continue;
            $vals[] = $ln;
        }
        return array_values(array_unique($vals));
    }

    // 4) Caminho local (TXT ou CSV)
    if (!is_readable($path)) throw new RuntimeException("Arquivo não legível: {$path}");

    $lower = strtolower($path);
    if (substr($lower, -4) === '.csv') {
        $h = fopen($path, 'r');
        if ($h === false) throw new RuntimeException("Falha ao abrir CSV: {$path}");
        $sepTry = ';';
        $header = fgetcsv($h, 0, $sepTry);
        if ($header === false) { fclose($h); throw new RuntimeException('CSV vazio.'); }
        if (count($header) === 1) { rewind($h); $sepTry = ','; $header = fgetcsv($h, 0, $sepTry); }
        $idx = null;
        foreach ($header as $i => $col) {
            if (strtolower(trim((string)$col)) === 'inscricao') { $idx = $i; break; }
        }
        if ($idx === null) { fclose($h); throw new RuntimeException('CSV precisa ter coluna "inscricao".'); }
        while (($row = fgetcsv($h, 0, $sepTry)) !== false) {
            $v = isset($row[$idx]) ? trim((string)$row[$idx]) : '';
            if ($v !== '') $vals[] = $v;
        }
        fclose($h);
    } else {
        $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        foreach ($lines as $ln) {
            $ln = trim($ln);
            if ($ln === '' || strpos($ln, '#') === 0 || strpos($ln, '//') === 0) continue;
            $vals[] = $ln;
        }
    }

    return array_values(array_unique($vals));
}


// ============================
// ENTRADAS (HTTP)
// ============================
$inputFile = null;
$listParam = null;

if (PHP_SAPI === 'cli') {
    $inputFile = $argv[1] ?? null;
} else {
    $inputFile = $_GET['file'] ?? null;
    $listParam = $_GET['list'] ?? null;
}

try {
    $inscricoes = load_inscricoes($inputFile, $listParam);
} catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    exit;
}

if (count($inscricoes) === 0) {
    echo json_encode(['status' => 'ok', 'processed' => 0, 'message' => 'Nenhuma inscrição fornecida.']);
    exit;
}

// ============================
// LOG
// ============================
$logDir = __DIR__ . DIRECTORY_SEPARATOR . 'logs';
if (!is_dir($logDir)) @mkdir($logDir, 0775, true);
$logPath = $logDir . DIRECTORY_SEPARATOR . 'sync_batch_' . date('Ymd_His') . '.ndjson';
$hLog = fopen($logPath, 'a');
function log_ndjson($h, array $obj) {
    fwrite($h, json_encode($obj, JSON_UNESCAPED_UNICODE) . PHP_EOL);
}

// ============================
// DB
// ============================
try {
    $pdo = new PDO($PG_DSN, $PG_USER, $PG_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);

    // prepares fixos
    $stmtDelInscricao = $pdo->prepare("DELETE FROM banco_el.banco_el_inscricao WHERE inscricao_municipal = :insc");
    $stmtInsInscricao = $pdo->prepare("
        INSERT INTO banco_el.banco_el_inscricao (
            inscricao_municipal, cep, tipo, codigo, numero, memorial, englobado,
            complemento, data_inclusao, data_exclusao, data_alteracao,
            exclusao_logica, processo_inclusao, processo_exclusao, correspondencia_local
        ) VALUES (
            :insc, :cep, :tipo, :codigo, :numero, :memorial, :englobado,
            :complemento, :data_inclusao, :data_exclusao, :data_alteracao,
            :exclusao_logica, :processo_inclusao, :processo_exclusao, :correspondencia_local
        )
    ");

    $stmtDelPessoa = $pdo->prepare("DELETE FROM banco_el.banco_el_pessoa WHERE inscricao_municipal = :insc");
    $stmtInsPessoa = $pdo->prepare("
        INSERT INTO banco_el.banco_el_pessoa (
            inscricao_municipal, id, nome, tipo, fantasia, documento, inscricao_estadual
        ) VALUES (
            :insc, :id, :nome, :tipo, :fantasia, :doc, :ie
        )
    ");

    $stmtDelLogr = $pdo->prepare("DELETE FROM banco_el.banco_el_logradouro WHERE inscricao_municipal = :insc");
    $stmtInsLogr = $pdo->prepare("
        INSERT INTO banco_el.banco_el_logradouro (
            inscricao_municipal, nome, tipo_id, tipo_sigla, tipo_descricao, codigo
        ) VALUES (
            :insc, :nome, :tipo_id, :sigla, :descricao, :codigo
        )
    ");

    $stmtDelBairro = $pdo->prepare("DELETE FROM banco_el.banco_el_bairro WHERE inscricao_municipal = :insc");
    $stmtInsBairro = $pdo->prepare("
        INSERT INTO banco_el.banco_el_bairro (
            inscricao_municipal, nome, codigo
        ) VALUES (
            :insc, :nome, :codigo
        )
    ");

    $stmtDelCidade = $pdo->prepare("DELETE FROM banco_el.banco_el_cidade WHERE inscricao_municipal = :insc");
    $stmtInsCidade = $pdo->prepare("
        INSERT INTO banco_el.banco_el_cidade (
            inscricao_municipal, nome, codigo_ibge, uf_sigla, uf_descricao
        ) VALUES (
            :insc, :nome, :ibge, :uf, :uf_desc
        )
    ");

    $stmtDelBoletim = $pdo->prepare("DELETE FROM banco_el.banco_el_boletim WHERE inscricao_municipal = :insc");
    $stmtInsBoletim = $pdo->prepare("
        INSERT INTO banco_el.banco_el_boletim (
            inscricao_municipal, campo, descricao, resposta_id, resposta_descricao
        ) VALUES (
            :insc, :campo, :descricao, :respid, :respdesc
        )
    ");

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Falha ao conectar no PostgreSQL: '.$e->getMessage()]);
    exit;
}

// ============================
// LOOP PRINCIPAL
// ============================
$total = count($inscricoes);
$ok = 0; $err = 0;

for ($i = 0; $i < $total; $i++) {
    $inscricao = trim($inscricoes[$i]);
    $tsStart = microtime(true);

    try {
        // 1) API
        $url = $API_URL . '?operacao=consulta&inscricao=' . urlencode($inscricao);
        $payload = curl_get_json($url);

        // 2) define chave efetiva
        $insc = get_n($payload, ['imobiliario', 'inscricaoCadastral']);
        if (!$insc) $insc = $inscricao;

        // 3) transação por inscrição
        $pdo->beginTransaction();

        // INSCRICAO
        $stmtDelInscricao->execute([':insc' => $insc]);
        $stmtInsInscricao->execute([
            ':insc'                => $insc,
            ':cep'                 => $payload['cep'] ?? null,
            ':tipo'                => $payload['tipo'] ?? null,
            ':codigo'              => isset($payload['codigo']) ? (int)$payload['codigo'] : null,
            ':numero'              => $payload['numero'] ?? null,
            ':memorial'            => $payload['memorial'] ?? null,
            ':englobado'           => $payload['englobado'] ?? null,
            ':complemento'         => $payload['complemento'] ?? null,
            ':data_inclusao'       => to_pg_ts($payload['dataInclusao'] ?? null),
            ':data_exclusao'       => to_pg_ts($payload['dataExclusao'] ?? null),
            ':data_alteracao'      => to_pg_ts($payload['dataAlteracao'] ?? null),
            ':exclusao_logica'     => $payload['exclusaoLogica'] ?? null,
            ':processo_inclusao'   => $payload['processoInclusao'] ?? null,
            ':processo_exclusao'   => $payload['processoExclusao'] ?? null,
            ':correspondencia_local' => $payload['correspondenciaLocal'] ?? null,
        ]);

        // PESSOA
        $stmtDelPessoa->execute([':insc' => $insc]);
        $p = $payload['pessoa'] ?? null;
        if (is_array($p)) {
            $stmtInsPessoa->execute([
                ':insc'     => $insc,
                ':id'       => $p['id'] ?? null,
                ':nome'     => $p['nome'] ?? null,
                ':tipo'     => $p['tipo'] ?? null,
                ':fantasia' => $p['fantasia'] ?? null,
                ':doc'      => $p['documento'] ?? null,
                ':ie'       => $p['inscricaoEstadual'] ?? null,
            ]);
        }

        // LOGRADOURO
        $stmtDelLogr->execute([':insc' => $insc]);
        $l = $payload['logradouro'] ?? null;
        $t = is_array($l) ? ($l['tipo'] ?? null) : null;
        if (is_array($l)) {
            $stmtInsLogr->execute([
                ':insc'      => $insc,
                ':nome'      => $l['nome'] ?? null,
                ':tipo_id'   => is_array($t) ? ($t['id'] ?? null) : null,
                ':sigla'     => is_array($t) ? ($t['sigla'] ?? null) : null,
                ':descricao' => is_array($t) ? ($t['descricao'] ?? null) : null,
                ':codigo'    => $l['codigo'] ?? null,
            ]);
        }

        // BAIRRO
        $stmtDelBairro->execute([':insc' => $insc]);
        $b = $payload['bairro'] ?? null;
        if (is_array($b)) {
            $stmtInsBairro->execute([
                ':insc'   => $insc,
                ':nome'   => $b['nome'] ?? null,
                ':codigo' => $b['codigo'] ?? null,
            ]);
        }

        // CIDADE
        $stmtDelCidade->execute([':insc' => $insc]);
        $c = $payload['cidade'] ?? null;
        $uf = is_array($c) ? ($c['tipo'] ?? null) : null;
        if (is_array($c)) {
            $stmtInsCidade->execute([
                ':insc'    => $insc,
                ':nome'    => $c['nome'] ?? null,
                ':ibge'    => $c['codigoIbge'] ?? null,
                ':uf'      => is_array($uf) ? ($uf['sigla'] ?? null) : null,
                ':uf_desc' => is_array($uf) ? ($uf['descricao'] ?? null) : null,
            ]);
        }

        // BOLETIM
        $stmtDelBoletim->execute([':insc' => $insc]);
        $blts = get_n($payload, ['imobiliario', 'boletim']);
        if (is_array($blts)) {
            foreach ($blts as $it) {
                $resp = is_array($it) ? ($it['resposta'] ?? null) : null;
                $stmtInsBoletim->execute([
                    ':insc'     => $insc,
                    ':campo'    => is_array($it) ? ($it['campo'] ?? null) : null,
                    ':descricao'=> is_array($it) ? ($it['descricao'] ?? null) : null,
                    ':respid'   => is_array($resp) ? ($resp['id'] ?? null) : null,
                    ':respdesc' => is_array($resp) ? ($resp['descricao'] ?? null) : null,
                ]);
            }
        }

        $pdo->commit();
        $ok++;
        log_ndjson($hLog, [
            'ts' => date('c'),
            'inscricao' => $insc,
            'status' => 'ok',
            'elapsed_ms' => (int)((microtime(true)-$tsStart)*1000)
        ]);

    } catch (Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        $err++;
        log_ndjson($hLog, [
            'ts' => date('c'),
            'inscricao' => $inscricao,
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }

    if ($DELAY_MS > 0) usleep($DELAY_MS * 1000);
}

if (is_resource($hLog)) fclose($hLog);
echo json_encode([
    'status' => 'ok',
    'processed' => $total,
    'success' => $ok,
    'errors' => $err,
    'log' => $logPath
], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
